<?php
/**
 * Simple JSON analytics logger (with external API geolocation).
 * Usage: require __DIR__ . '/analytics.php';
 * Logs to logs/visits-YYYY-MM-DD.jsonl (one JSON object per line).
 */

////////////////////
// Configuration  //
////////////////////
$ANA = [
    'log_dir'        => __DIR__ . '/logs',
    'filename_fmt'   => 'visits-%Y-%m-%d.jsonl',
    'respect_dnt'    => true,
    'skip_bots'      => true,
    'ip_anonymize'   => true, // keep this true for privacy
    'extra_fields'   => [],

    // Geo via external API (no DB)
    'geo_enabled'    => true,
    'geo_provider'   => 'ipwho', // 'ipwho' | 'ipapi' (see lookup_country_via_api)
    'geo_timeout'    => 1.5,     // seconds
    'geo_ttl'        => 86400 * 7, // cache: 7 days
    'geo_cache_file' => __DIR__ . '/logs/geo-cache.json',
    'geo_cache_salt' => 'change-this-secret-salt', // used to hash IPs for cache keys
];

////////////////////
// Implementation //
////////////////////
if (PHP_SAPI !== 'cli') {
    try {
        if (!isset($GLOBALS['DISABLE_SIMPLE_ANALYTICS']) || !$GLOBALS['DISABLE_SIMPLE_ANALYTICS']) {
            if (!$ANA['respect_dnt'] || (isset($_SERVER['HTTP_DNT']) && $_SERVER['HTTP_DNT'] === '1') === false) {
                if (!$ANA['skip_bots'] || !is_bot()) {
                    log_visit();
                }
            }
        }
    } catch (Throwable $e) {
        // Fail silently
        // error_log('[analytics] ' . $e->getMessage());
    }
}

function log_visit(): void {
    global $ANA;

    if (!is_dir($ANA['log_dir'])) {
        @mkdir($ANA['log_dir'], 0775, true);
    }

    $path = $ANA['log_dir'] . '/' . strftime($ANA['filename_fmt']);
    $ip   = get_ip();

    // Country detection BEFORE anonymizing (external API, with salted-hash cache)
    $country = null;
    if ($ANA['geo_enabled'] && $ip && filter_var($ip, FILTER_VALIDATE_IP)) {
        $country = geo_country_cached($ip);
    }

    $now = new DateTimeImmutable('now', new DateTimeZone('UTC'));
    $record = array_merge([
        'ts'      => $now->format(DateTime::ATOM),
        'method'  => $_SERVER['REQUEST_METHOD'] ?? null,
        'scheme'  => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http',
        'host'    => $_SERVER['HTTP_HOST'] ?? null,
        'path'    => parse_url(($_SERVER['REQUEST_URI'] ?? ''), PHP_URL_PATH),
        'query'   => ($_SERVER['QUERY_STRING'] ?? '') ?: null,
        'ref'     => $_SERVER['HTTP_REFERER'] ?? null,
        'ua'      => $_SERVER['HTTP_USER_AGENT'] ?? null,
        'lang'    => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? null,
        'ip'      => anonymize_ip($ip),
        'country' => $country, // 2-letter ISO code (e.g. "BG") or null
    ], $ANA['extra_fields']);

    $line = json_encode($record, JSON_UNESCAPED_SLASHES) . PHP_EOL;
    $fh = @fopen($path, 'ab');
    if ($fh === false) return;
    if (flock($fh, LOCK_EX)) {
        fwrite($fh, $line);
        fflush($fh);
        flock($fh, LOCK_UN);
    }
    fclose($fh);
}

////////////////////
// Geo helpers    //
////////////////////
function geo_country_cached(string $ip): ?string {
    global $ANA;

    $cacheFile = $ANA['geo_cache_file'];
    $ttl       = (int)$ANA['geo_ttl'];
    $salt      = (string)$ANA['geo_cache_salt'];
    $key       = sha1($salt . '|' . $ip);

    // Load cache (best effort)
    $cache = [];
    if (is_file($cacheFile)) {
        $raw = @file_get_contents($cacheFile);
        if ($raw !== false) {
            $tmp = json_decode($raw, true);
            if (is_array($tmp)) $cache = $tmp;
        }
    }

    // Hit & fresh?
    if (isset($cache[$key])) {
        $row = $cache[$key];
        if (isset($row['t'], $row['c']) && time() - (int)$row['t'] < $ttl) {
            return $row['c'] ?: null;
        }
    }

    // Miss -> fetch
    $country = lookup_country_via_api($ip);

    // Update cache (do not store raw IP)
    $cache[$key] = ['c' => $country, 't' => time()];
    // Best-effort write
    @file_put_contents($cacheFile, json_encode($cache, JSON_UNESCAPED_SLASHES), LOCK_EX);

    return $country;
}

function lookup_country_via_api(string $ip): ?string {
    global $ANA;

    $timeout = (float)$ANA['geo_timeout'];
    $provider = $ANA['geo_provider'];

    $ctx = stream_context_create([
        'http' => [
            'method'  => 'GET',
            'timeout' => $timeout,
            'header'  => "User-Agent: SimpleAnalytics/1.0\r\nAccept: application/json\r\n",
        ],
        'ssl' => [
            'verify_peer'      => true,
            'verify_peer_name' => true,
        ],
    ]);

    try {
        if ($provider === 'ipwho') {
            // https://ipwho.is/{ip}
            $url = 'https://ipwho.is/' . rawurlencode($ip);
            $json = @file_get_contents($url, false, $ctx);
            if ($json === false) return null;
            $data = json_decode($json, true);
            if (is_array($data) && ($data['success'] ?? false)) {
                return $data['country_code'] ?? null; // e.g. "US", "BG"
            }
            return null;
        } elseif ($provider === 'ipapi') {
            // https://ipapi.co/{ip}/json/
            $url = 'https://ipapi.co/' . rawurlencode($ip) . '/json/';
            $json = @file_get_contents($url, false, $ctx);
            if ($json === false) return null;
            $data = json_decode($json, true);
            if (is_array($data)) {
                return $data['country_code'] ?? null;
            }
            return null;
        } else {
            return null;
        }
    } catch (Throwable $e) {
        return null;
    }
}

////////////////////
// Misc helpers   //
////////////////////
function get_ip(): ?string {
    // Use common headers cautiously; prefer the first valid IP in XFF
    $keys = [
        'HTTP_CF_CONNECTING_IP', // Cloudflare
        'HTTP_X_REAL_IP',
        'HTTP_X_FORWARDED_FOR',
        'REMOTE_ADDR',
    ];
    foreach ($keys as $k) {
        if (!empty($_SERVER[$k])) {
            $val = $_SERVER[$k];
            if ($k === 'HTTP_X_FORWARDED_FOR') {
                foreach (explode(',', $val) as $part) {
                    $ip = trim($part);
                    if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
                }
            }
            if (filter_var($val, FILTER_VALIDATE_IP)) return $val;
        }
    }
    return null;
}

function anonymize_ip(?string $ip): ?string {
    global $ANA;
    if (!$ANA['ip_anonymize'] || !$ip) return $ip;
    if (strpos($ip, ':') !== false) { // IPv6
        // zero last 80 bits (~ /48 kept)
        $parts = explode(':', $ip);
        $parts = array_pad($parts, 8, '0');
        for ($i = 3; $i < 8; $i++) $parts[$i] = '0000';
        return implode(':', $parts);
    }
    // IPv4
    $oct = explode('.', $ip);
    if (count($oct) === 4) { $oct[3] = '0'; return implode('.', $oct); }
    return $ip;
}

function is_bot(): bool {
    $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
    if ($ua === '') return true;
    $needles = [
        'bot','crawl','spider','slurp','mediapartners','facebookexternalhit','pinterest','scanner',
        'curl','wget','python-requests','libwww-perl','httpclient','feedfetcher','validator','node-fetch',
        'datadog','newrelic','monitor','uptime','statuscake','headless','selenium','playwright'
    ];
    $ua_l = strtolower($ua);
    foreach ($needles as $n) {
        if (strpos($ua_l, $n) !== false) return true;
    }
    return false;
}
