<?php
// Start session for CSRF protection
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once 'config/database.php';
require_once 'config/whatsapp.php';
require_once 'includes/Validator.php';
require_once 'includes/ErrorHandler.php';
require_once 'includes/EmailService.php';

// Initialize error handling
ErrorHandler::init();

// Initialize variables
$errors = [];
$success = false;

try {
    if (isset($_POST['contact_name'])) {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        // Validate and sanitize input
        $name = Validator::validateText($_POST['contact_name'] ?? '', 100);
        $email = Validator::validateEmail($_POST['contact_email'] ?? '');
        $phone = Validator::validateText($_POST['contact_phone'] ?? '', 20);
        $subject = Validator::validateText($_POST['contact_subject'] ?? '', 100);
        $message = Validator::validateText($_POST['contact_message'] ?? '', 1000);
        
        // Check for validation errors
        if (!$name) $errors[] = "Please enter your name";
        if (!$email) $errors[] = "Please enter a valid email address";
        if (!$subject) $errors[] = "Please enter a subject";
        if (!$message) $errors[] = "Please enter your message";
        
        // If no validation errors, proceed with email sending
        if (empty($errors)) {
            $contactData = [
                'name' => $name,
                'email' => $email,
                'phone' => $phone ?: 'Not provided',
                'subject' => $subject,
                'message' => $message
            ];
            
            $emailSent = EmailService::sendContactForm($contactData);
            
            if ($emailSent) {
                $success = true;
            } else {
                $errors[] = "Failed to send message. Please try again later.";
            }
        }
    }
} catch (Exception $e) {
    error_log("Contact form error: " . $e->getMessage());
    $errors[] = "An error occurred. Please try again later.";
}

// Generate CSRF token for form
$csrfToken = Validator::generateCSRFToken();

include("nav.php");
?>

<!doctype html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Content-Security-Policy" content="upgrade-insecure-requests">
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="assets/favicon/favicon.ico">
    <link rel="icon" type="image/png" sizes="16x16" href="assets/favicon/favicon-16x16.png">
    <link rel="icon" type="image/png" sizes="32x32" href="assets/favicon/favicon-32x32.png">
    <link rel="apple-touch-icon" sizes="180x180" href="assets/favicon/apple-touch-icon.png">
    <link rel="manifest" href="assets/favicon/site.webmanifest">
    
    <!-- Tailwind CSS -->
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    
    <title>Contact Us - Migration.ng</title>
</head>
<body class="bg-gray-50 dark:bg-gray-900">
    
    <!-- Display error messages -->
    <?php if (!empty($errors)): ?>
        <div class="fixed top-4 right-4 z-50 max-w-sm">
            <?php foreach ($errors as $error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-2 shadow-lg">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <svg class="h-5 w-5 text-red-400" viewBox="0 0 20 20" fill="currentColor">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd" />
                            </svg>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm"><?php echo htmlspecialchars($error); ?></p>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
    
    <!-- Display success message -->
    <?php if ($success): ?>
        <div class="fixed top-4 right-4 z-50 max-w-sm">
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded shadow-lg">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <svg class="h-5 w-5 text-green-400" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd" />
                        </svg>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm">Message sent successfully! We'll get back to you within 24 hours.</p>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <!-- Hero Section -->
    <section class="bg-gradient-to-br from-blue-600 via-blue-700 to-indigo-800 text-white">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16 lg:py-24">
            <div class="text-center">
                <h1 class="text-4xl md:text-5xl lg:text-6xl font-bold mb-6">
                    Get in Touch
                </h1>
                <p class="text-xl md:text-2xl text-blue-100 max-w-3xl mx-auto">
                    Have questions about our services? We're here to help you with your migration journey.
                </p>
            </div>
        </div>
    </section>

    <!-- Contact Section -->
    <section class="py-16 lg:py-24">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-12 lg:gap-16">
                
                <!-- Contact Information -->
                <div class="space-y-8">
                    <div>
                        <h2 class="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                            Let's Start a Conversation
                        </h2>
                        <p class="text-lg text-gray-600 dark:text-gray-300 mb-8">
                            Whether you have questions about our services, need pricing information, or want to discuss your specific migration needs, we're here to help.
                        </p>
                    </div>

                    <!-- Contact Cards -->
                    <div class="space-y-6">
                        <!-- Email -->
                        <div class="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-lg border border-gray-200 dark:border-gray-700">
                            <div class="flex items-center">
                                <div class="flex-shrink-0">
                                    <div class="w-12 h-12 bg-blue-100 dark:bg-blue-900 rounded-lg flex items-center justify-center">
                                        <svg class="w-6 h-6 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 4.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"></path>
                                        </svg>
                                    </div>
                                </div>
                                <div class="ml-4">
                                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Email Us</h3>
                                    <p class="text-gray-600 dark:text-gray-300">services@migration.ng</p>
                                    <p class="text-sm text-gray-500 dark:text-gray-400">We typically respond within 24 hours</p>
                                </div>
                            </div>
                        </div>

                        <!-- WhatsApp -->
                        <div class="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-lg border border-gray-200 dark:border-gray-700">
                            <div class="flex items-center">
                                <div class="flex-shrink-0">
                                    <div class="w-12 h-12 bg-green-100 dark:bg-green-900 rounded-lg flex items-center justify-center">
                                        <i class="fab fa-whatsapp text-2xl text-green-600 dark:text-green-400"></i>
                                    </div>
                                </div>
                                <div class="ml-4">
                                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white">WhatsApp Us</h3>
                                    <a href="<?php echo WHATSAPP_LINK; ?>" target="_blank" class="text-green-600 dark:text-green-400 hover:text-green-700 dark:hover:text-green-300 font-medium"><?php echo WHATSAPP_DISPLAY_TEXT; ?></a>
                                    <p class="text-sm text-gray-500 dark:text-gray-400">Get instant responses during business hours</p>
                                </div>
                            </div>
                        </div>

                        <!-- Location -->
                        <div class="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-lg border border-gray-200 dark:border-gray-700">
                            <div class="flex items-center">
                                <div class="flex-shrink-0">
                                    <div class="w-12 h-12 bg-green-100 dark:bg-green-900 rounded-lg flex items-center justify-center">
                                        <svg class="w-6 h-6 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"></path>
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                        </svg>
                                    </div>
                                </div>
                                <div class="ml-4">
                                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Our Office</h3>
                                    <p class="text-gray-600 dark:text-gray-300">Abuja, Nigeria</p>
                                    <p class="text-sm text-gray-500 dark:text-gray-400">Serving clients worldwide</p>
                                </div>
                            </div>
                        </div>

                        <!-- Hours -->
                        <div class="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-lg border border-gray-200 dark:border-gray-700">
                            <div class="flex items-center">
                                <div class="flex-shrink-0">
                                    <div class="w-12 h-12 bg-purple-100 dark:bg-purple-900 rounded-lg flex items-center justify-center">
                                        <svg class="w-6 h-6 text-purple-600 dark:text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                        </svg>
                                    </div>
                                </div>
                                <div class="ml-4">
                                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Business Hours</h3>
                                    <p class="text-gray-600 dark:text-gray-300">Monday - Friday: 9AM - 6PM</p>
                                    <p class="text-sm text-gray-500 dark:text-gray-400">WAT (West African Time)</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- FAQ Link -->
                    <div class="bg-blue-50 dark:bg-blue-900/20 rounded-xl p-6 border border-blue-200 dark:border-blue-700">
                        <h3 class="text-lg font-semibold text-blue-900 dark:text-blue-100 mb-2">
                            Have Common Questions?
                        </h3>
                        <p class="text-blue-700 dark:text-blue-300 mb-4">
                            Check out our frequently asked questions for quick answers.
                        </p>
                        <a href="/about.php" class="inline-flex items-center text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 font-medium">
                            View FAQ
                            <svg class="ml-2 w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                            </svg>
                        </a>
                    </div>
                </div>

                <!-- Contact Form -->
                <div class="bg-white dark:bg-gray-800 rounded-2xl shadow-xl border border-gray-200 dark:border-gray-700 p-8">
                    <div class="mb-8">
                        <h2 class="text-2xl font-bold text-gray-900 dark:text-white mb-2">
                            Send us a Message
                        </h2>
                        <p class="text-gray-600 dark:text-gray-300">
                            Fill out the form below and we'll get back to you as soon as possible.
                        </p>
                    </div>

                    <form method="POST" action="" class="space-y-6">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                        
                        <!-- Name -->
                        <div>
                            <label for="contact_name" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Full Name *
                            </label>
                            <input
                                type="text"
                                name="contact_name"
                                id="contact_name"
                                required
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white dark:placeholder-gray-400 transition-colors"
                                placeholder="Enter your full name"
                                value="<?php echo htmlspecialchars($_POST['contact_name'] ?? ''); ?>"
                            />
                        </div>

                        <!-- Email -->
                        <div>
                            <label for="contact_email" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Email Address *
                            </label>
                            <input
                                type="email"
                                name="contact_email"
                                id="contact_email"
                                required
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white dark:placeholder-gray-400 transition-colors"
                                placeholder="Enter your email address"
                                value="<?php echo htmlspecialchars($_POST['contact_email'] ?? ''); ?>"
                            />
                        </div>

                        <!-- Phone -->
                        <div>
                            <label for="contact_phone" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Phone Number
                            </label>
                            <input
                                type="tel"
                                name="contact_phone"
                                id="contact_phone"
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white dark:placeholder-gray-400 transition-colors"
                                placeholder="Enter your phone number (optional)"
                                value="<?php echo htmlspecialchars($_POST['contact_phone'] ?? ''); ?>"
                            />
                        </div>

                        <!-- Subject -->
                        <div>
                            <label for="contact_subject" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Subject *
                            </label>
                            <input
                                type="text"
                                name="contact_subject"
                                id="contact_subject"
                                required
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white dark:placeholder-gray-400 transition-colors"
                                placeholder="What is this about?"
                                value="<?php echo htmlspecialchars($_POST['contact_subject'] ?? ''); ?>"
                            />
                        </div>

                        <!-- Message -->
                        <div>
                            <label for="contact_message" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Message *
                            </label>
                            <textarea
                                name="contact_message"
                                id="contact_message"
                                rows="6"
                                required
                                class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white dark:placeholder-gray-400 transition-colors resize-none"
                                placeholder="Tell us about your inquiry or questions..."
                            ><?php echo htmlspecialchars($_POST['contact_message'] ?? ''); ?></textarea>
                        </div>

                        <!-- Submit Button -->
                        <div>
                            <button
                                type="submit"
                                class="w-full bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white font-semibold py-3 px-6 rounded-lg transition-all duration-300 transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
                            >
                                <span class="flex items-center justify-center">
                                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"></path>
                                    </svg>
                                    Send Message
                                </span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="bg-gray-100 dark:bg-gray-800 py-16">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
            <h2 class="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                Ready to Start Your Application?
            </h2>
            <p class="text-lg text-gray-600 dark:text-gray-300 mb-8 max-w-2xl mx-auto">
                If you're ready to begin your migration journey, access our portal to start your application process.
            </p>
            <a href="/portal/login.php" class="inline-flex items-center px-8 py-3 text-lg font-medium text-white bg-blue-600 hover:bg-blue-700 rounded-lg transition-colors duration-300">
                <svg class="w-5 h-5 mr-2" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M3 4a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1zm0 4a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1zm0 4a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1z" clip-rule="evenodd"></path>
                </svg>
                Access Portal
            </a>
        </div>
    </section>

    <!-- Floating WhatsApp Button -->
    <a href="<?php echo WHATSAPP_LINK; ?>" target="_blank" class="fixed bottom-6 right-6 z-50 bg-green-500 hover:bg-green-600 text-white rounded-full p-4 shadow-lg transition-all duration-300 transform hover:scale-110 hover:shadow-xl">
        <i class="fab fa-whatsapp text-2xl"></i>
    </a>

</body>
<?php include 'footer.php'; ?>
</html>