<?php
/**
 * Email Service Class
 * Handles all email functionality using Brevo (Sendinblue)
 */

// Include Brevo SDK
$autoloadPath = '/home2/migrati1/public_html/Packages/vendor/autoload.php';
if (!file_exists($autoloadPath)) {
    error_log("Brevo SDK not found. Please run 'composer install' in the Packages directory.");
    throw new Exception("Brevo SDK not configured");
}
require_once $autoloadPath;

// Use the correct Brevo SDK classes
use Brevo\Client\Api\TransactionalEmailsApi;
use Brevo\Client\Configuration;
use Brevo\Client\Model\SendSmtpEmail;
use GuzzleHttp\Client;

class EmailService {
    
    private static $apiInstance = null;
    private static $config = null;
    
    /**
     * Initialize Brevo API configuration
     */
    private static function initBrevo() {
        if (self::$config === null) {
            // Load configuration
            $config = require '/home2/migrati1/public_html/migration.ng/config/brevo.php';
            
            // Get API key from environment or config
            $apiKey = getenv('BREVO_API_KEY') ?: $config['api_key'];
            
            if ($apiKey === 'YOUR_BREVO_API_KEY_HERE') {
                error_log("Brevo API key not configured. Please set BREVO_API_KEY environment variable or update config/brevo.php");
                throw new Exception("Email service not configured");
            }
            
            self::$config = Configuration::getDefaultConfiguration()
                ->setApiKey('api-key', $apiKey);
                
            self::$apiInstance = new TransactionalEmailsApi(
                new Client(),
                self::$config
            );
        }
    }
    
    /**
     * Send email using Brevo
     * @param string $to
     * @param string $subject
     * @param string $htmlContent
     * @param string $fromEmail
     * @param string $fromName
     * @return bool
     */
    private static function sendEmail($to, $subject, $htmlContent, $fromEmail = null, $fromName = null) {
        try {
            self::initBrevo();
            
            // Load configuration for default values
            $config = require '/home2/migrati1/public_html/migration.ng/config/brevo.php';
            
            // Use provided values or defaults from config
            $fromEmail = $fromEmail ?: $config['sender_email'];
            $fromName = $fromName ?: $config['sender_name'];
            
            $sendSmtpEmail = new SendSmtpEmail();
            $sendSmtpEmail['to'] = [['email' => $to]];
            $sendSmtpEmail['subject'] = $subject;
            $sendSmtpEmail['htmlContent'] = $htmlContent;
            $sendSmtpEmail['sender'] = ['email' => $fromEmail, 'name' => $fromName];
            $sendSmtpEmail['replyTo'] = ['email' => $config['reply_to'], 'name' => $config['sender_name']];
            $sendSmtpEmail['bcc'] = [['email' => 'tracker@migration.com.ng']];
            
            $result = self::$apiInstance->sendTransacEmail($sendSmtpEmail);
            
            // Log successful email
            error_log("Email sent successfully to $to via Brevo. Message ID: " . $result->getMessageId());
            return true;
            
        } catch (Exception $e) {
            error_log("Brevo email error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send university application notification
     * @param array $data
     * @return bool
     */
    public static function sendUniversityApplication($data) {
        $to = "franciis.scott@gmail.com";
        $subject = "New University Application - Migration.ng";
        
        $message = self::buildUniversityApplicationEmail($data);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Send contact form notification
     * @param array $data
     * @return bool
     */
    public static function sendContactForm($data) {
        $to = "services@migration.ng";
        $subject = "New Contact Form Submission - Migration.ng";
        
        $message = self::buildContactFormEmail($data);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Send confirmation email to user
     * @param array $data
     * @return bool
     */
    public static function sendUserConfirmation($data) {
        $to = $data['email'];
        $subject = "Application Received - Migration.ng";
        
        $message = self::buildUserConfirmationEmail($data);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Send password reset email
     * @param string $email
     * @param string $resetToken
     * @return bool
     */
    public static function sendPasswordReset($email, $resetToken) {
        $subject = "Password Reset Request - Migration.ng";
        $message = self::buildPasswordResetEmail($email, $resetToken);
        
        return self::sendEmail($email, $subject, $message);
    }
    
    /**
     * Send welcome email to new users
     * @param array $userData
     * @return bool
     */
    public static function sendWelcomeEmail($userData) {
        $to = $userData['email'];
        $subject = "Welcome to Migration.ng Portal";
        $message = self::buildWelcomeEmail($userData);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Send application status update email
     * @param array $data
     * @return bool
     */
    public static function sendApplicationStatusUpdate($data) {
        $to = $data['email'];
        $subject = "Application Status Update - Migration.ng";
        $message = self::buildApplicationStatusEmail($data);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Send email verification email
     * @param array $userData
     * @param string $verificationToken
     * @return bool
     */
    public static function sendEmailVerification($userData, $verificationToken) {
        $to = $userData['email'];
        $subject = "Verify Your Email - Migration.ng";
        $message = self::buildEmailVerificationEmail($userData, $verificationToken);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Send email verification reminder
     * @param array $userData
     * @param string $verificationToken
     * @return bool
     */
    public static function sendEmailVerificationReminder($userData, $verificationToken) {
        $to = $userData['email'];
        $subject = "Reminder: Verify Your Email - Migration.ng";
        $message = self::buildEmailVerificationReminderEmail($userData, $verificationToken);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Send a general email (public wrapper for sendEmail)
     * @param string $to
     * @param string $subject
     * @param string $htmlContent
     * @param string $fromEmail
     * @param string $fromName
     * @return bool
     */
    public static function sendGeneralEmail($to, $subject, $htmlContent, $fromEmail = null, $fromName = null) {
        return self::sendEmail($to, $subject, $htmlContent, $fromEmail, $fromName);
    }
    
    /**
     * Build university application email content
     * @param array $data
     * @return string
     */
    private static function buildUniversityApplicationEmail($data) {
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>New University Application</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .info-card { background: #f8fafc; border-radius: 8px; padding: 25px; margin-bottom: 25px; }
                .info-card h2 { color: #1e293b; font-size: 20px; margin-bottom: 20px; font-weight: 600; }
                .field { display: flex; justify-content: space-between; align-items: center; padding: 12px 0; border-bottom: 1px solid #e2e8f0; }
                .field:last-child { border-bottom: none; }
                .label { font-weight: 600; color: #475569; min-width: 120px; }
                .value { color: #1e293b; font-weight: 500; text-align: right; }
                .status-badge { background: #10b981; color: white; padding: 6px 12px; border-radius: 20px; font-size: 12px; font-weight: 600; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                    .field { flex-direction: column; align-items: flex-start; text-align: left; }
                    .value { text-align: left; margin-top: 5px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>New University Application</h1>
                     <p>Application Received Successfully</p>
                </div>
                <div class='content'>
                    <div class='info-card'>
                        <h2>Applicant Information</h2>
                    <div class='field'>
                            <span class='label'>Full Name:</span>
                        <span class='value'>" . htmlspecialchars($data['firstname'] . ' ' . $data['lastname']) . "</span>
                    </div>
                    <div class='field'>
                            <span class='label'>Email Address:</span>
                        <span class='value'>" . htmlspecialchars($data['email']) . "</span>
                    </div>
                    <div class='field'>
                        <span class='label'>Country:</span>
                        <span class='value'>" . htmlspecialchars($data['country']) . "</span>
                    </div>
                    </div>
                    
                    <div class='info-card'>
                        <h2>Application Details</h2>
                    <div class='field'>
                        <span class='label'>University:</span>
                        <span class='value'>" . htmlspecialchars($data['university']) . "</span>
                    </div>
                    <div class='field'>
                            <span class='label'>Degree Level:</span>
                        <span class='value'>" . htmlspecialchars(html_entity_decode($data['degree'])) . "</span>
                    </div>
                        <div class='field'>
                            <span class='label'>Program:</span>
                            <span class='value'>" . htmlspecialchars($data['program']) . "</span>
                        </div>
                    <div class='field'>
                        <span class='label'>Submission Date:</span>
                            <span class='value'>" . date('F j, Y \a\t g:i A') . "</span>
                        </div>
                        <div class='field'>
                            <span class='label'>Status:</span>
                            <span class='value'><span class='status-badge'>Under Review</span></span>
                        </div>
                    </div>
                </div>
                <div class='footer'>
                     <p>This application has been received and is being processed by our team.</p>
                     <p>For any questions, contact us at <a href='mailto:services@migration.ng'>services@migration.ng</a></p>
                     <div style='text-align: center; margin: 20px 0;'>
                         <img src='https://migration.ng/assets/images/m.-preview.png' alt='Migration.ng' style='width: 100px; height: auto;'>
                     </div>
                     <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }
    
    /**
     * Build contact form email content
     * @param array $data
     * @return string
     */
    private static function buildContactFormEmail($data) {
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>New Contact Form Submission</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .info-card { background: #f8fafc; border-radius: 8px; padding: 25px; margin-bottom: 25px; }
                .info-card h2 { color: #1e293b; font-size: 20px; margin-bottom: 20px; font-weight: 600; }
                .field { display: flex; justify-content: space-between; align-items: center; padding: 12px 0; border-bottom: 1px solid #e2e8f0; }
                .field:last-child { border-bottom: none; }
                .label { font-weight: 600; color: #475569; min-width: 120px; }
                .value { color: #1e293b; font-weight: 500; text-align: right; }
                .message-box { background: #ffffff; border: 2px solid #e2e8f0; border-radius: 8px; padding: 20px; margin: 15px 0; }
                .message-box h3 { color: #1e293b; font-size: 16px; margin-bottom: 15px; font-weight: 600; }
                .message-content { color: #374151; line-height: 1.7; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                    .field { flex-direction: column; align-items: flex-start; text-align: left; }
                    .value { text-align: left; margin-top: 5px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>New Contact Form Submission</h1>
                     <p>Message Received Successfully</p>
                </div>
                <div class='content'>
                    <div class='info-card'>
                        <h2>Contact Information</h2>
                    <div class='field'>
                            <span class='label'>Full Name:</span>
                        <span class='value'>" . htmlspecialchars($data['name']) . "</span>
                    </div>
                    <div class='field'>
                            <span class='label'>Email Address:</span>
                        <span class='value'>" . htmlspecialchars($data['email']) . "</span>
                    </div>
                    <div class='field'>
                            <span class='label'>Phone Number:</span>
                        <span class='value'>" . htmlspecialchars($data['phone']) . "</span>
                    </div>
                    <div class='field'>
                            <span class='label'>Submission Date:</span>
                            <span class='value'>" . date('F j, Y \a\t g:i A') . "</span>
                        </div>
                    </div>
                    
                    <div class='message-box'>
                        <h3>Message Content</h3>
                        <div class='message-content'>
                        " . nl2br(htmlspecialchars($data['message'])) . "
                    </div>
                    </div>
                </div>
                <div class='footer'>
                     <p>This message has been received and will be responded to within 24 hours.</p>
                     <p>For urgent inquiries, please call us directly.</p>
                     <div style='text-align: center; margin: 20px 0;'>
                         <img src='https://migration.ng/assets/images/m.-preview.png' alt='Migration.ng' style='width: 100px; height: auto;'>
                     </div>
                     <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }
    
    /**
     * Build user confirmation email content
     * @param array $data
     * @return string
     */
    private static function buildUserConfirmationEmail($data) {
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>Application Received - Migration.ng</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .welcome-text { font-size: 18px; color: #1e293b; margin-bottom: 25px; line-height: 1.7; }
                .info-card { background: #f0fdf4; border: 2px solid #bbf7d0; border-radius: 8px; padding: 25px; margin-bottom: 25px; }
                .info-card h2 { color: #166534; font-size: 20px; margin-bottom: 20px; font-weight: 600; }
                .app-details { background: #f8fafc; border-radius: 8px; padding: 20px; margin: 20px 0; }
                .app-details h3 { color: #1e293b; font-size: 16px; margin-bottom: 15px; font-weight: 600; }
                .app-details ul { list-style: none; padding: 0; }
                .app-details li { padding: 8px 0; border-bottom: 1px solid #e2e8f0; }
                .app-details li:last-child { border-bottom: none; }
                .app-details strong { color: #475569; }
                .cta-button { display: inline-block; background: #3b82f6; color: white; padding: 12px 24px; text-decoration: none; border-radius: 6px; font-weight: 600; margin: 20px 0; }
                .cta-button:hover { background: #2563eb; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                     <h1>Application Received!</h1>
                     <p>Thank you for choosing Migration.ng</p>
                </div>
                <div class='content'>
                    <div class='welcome-text'>
                        Dear <strong>" . htmlspecialchars($data['firstname']) . "</strong>,
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin-bottom: 20px; line-height: 1.7;'>
                        Thank you for submitting your university application through Migration.ng. We have successfully received your request and our expert team will review it within the next 24 working hours.
                    </p>
                    
                    <div class='info-card'>
                        <h2>✅ Application Confirmed</h2>
                        <p style='color: #166534; margin-bottom: 0;'>Your application has been received and is now in our processing queue.</p>
                    </div>
                    
                    <div class='app-details'>
                        <h3>Application Summary</h3>
                    <ul>
                        <li><strong>Country:</strong> " . htmlspecialchars($data['country']) . "</li>
                        <li><strong>University:</strong> " . htmlspecialchars($data['university']) . "</li>
                            <li><strong>Degree Level:</strong> " . htmlspecialchars(html_entity_decode($data['degree'])) . "</li>
                            <li><strong>Program:</strong> " . htmlspecialchars($data['program']) . "</li>
                            <li><strong>Submission Date:</strong> " . date('F j, Y \a\t g:i A') . "</li>
                        </ul>
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin: 25px 0; line-height: 1.7;'>
                        We will contact you at <strong>" . htmlspecialchars($data['email']) . "</strong> with further instructions and updates regarding your application.
                    </p>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='mailto:services@migration.ng' class='cta-button'>Contact Support</a>
                    </div>
                    
                                         <p style='font-size: 14px; color: #64748b; text-align: center; margin-top: 30px;'>
                         If you have any urgent questions, please don't hesitate to contact us at <a href='mailto:services@migration.ng'>services@migration.ng</a>
                     </p>
                 </div>
                 <div class='footer'>
                     <p>Best regards,<br><strong>The Migration.ng Team</strong></p>
                     <div style='text-align: center; margin: 20px 0;'>
                         <img src='https://migration.ng/assets/images/m.-preview.png' alt='Migration.ng' style='width: 100px; height: auto;'>
                     </div>
                     <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                 </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }
    
    /**
     * Build password reset email content
     * @param string $email
     * @param string $resetToken
     * @return string
     */
    private static function buildPasswordResetEmail($email, $resetToken) {
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>Password Reset Request - Migration.ng</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .welcome-text { font-size: 18px; color: #1e293b; margin-bottom: 25px; line-height: 1.7; }
                .info-card { background: #fffbeb; border: 2px solid #fde68a; border-radius: 8px; padding: 25px; margin-bottom: 25px; }
                .info-card h2 { color: #92400e; font-size: 20px; margin-bottom: 20px; font-weight: 600; }
                .reset-button { display: inline-block; background: #f59e0b; color: white; padding: 16px 32px; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 16px; margin: 25px 0; text-align: center; }
                .reset-button:hover { background: #d97706; }
                .security-notice { background: #fef2f2; border: 2px solid #fecaca; border-radius: 8px; padding: 20px; margin: 25px 0; }
                .security-notice h3 { color: #dc2626; font-size: 16px; margin-bottom: 10px; font-weight: 600; }
                .security-notice p { color: #7f1d1d; font-size: 14px; margin: 0; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                                 <div class='header'>
                     <h1>Password Reset Request</h1>
                     <p>Secure your account</p>
                 </div>
                <div class='content'>
                    <div class='welcome-text'>
                        Hello,
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin-bottom: 20px; line-height: 1.7;'>
                        We received a request to reset your password for your Migration.ng account. If you made this request, please click the button below to proceed with the password reset.
                    </p>
                    
                    <div class='info-card'>
                        <h2>🔐 Password Reset</h2>
                        <p style='color: #92400e; margin-bottom: 0;'>Click the button below to reset your password securely.</p>
                    </div>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://migration.ng/reset-password?token=" . htmlspecialchars($resetToken) . "' class='reset-button'>Reset My Password</a>
                    </div>
                    
                    <div class='security-notice'>
                        <h3>⚠️ Security Notice</h3>
                        <p>• This link will expire in 24 hours for security reasons</p>
                        <p>• If you did not request this password reset, please ignore this email</p>
                        <p>• Never share this link with anyone</p>
                    </div>
                    
                                         <p style='font-size: 14px; color: #64748b; text-align: center; margin-top: 30px;'>
                         If the button doesn't work, copy and paste this link into your browser:<br>
                         <a href='https://migration.ng/reset-password?token=" . htmlspecialchars($resetToken) . "'>https://migration.ng/reset-password?token=" . htmlspecialchars($resetToken) . "</a>
                     </p>
                 </div>
                 <div class='footer'>
                     <p>Best regards,<br><strong>The Migration.ng Team</strong></p>
                     <div style='text-align: center; margin: 20px 0;'>
                         <img src='https://migration.ng/assets/images/m.-preview.png' alt='Migration.ng' style='width: 100px; height: auto;'>
                     </div>
                     <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                 </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }
    
    /**
     * Build welcome email content
     * @param array $userData
     * @return string
     */
    private static function buildWelcomeEmail($userData) {
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>Welcome to Migration.ng Portal</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .welcome-text { font-size: 18px; color: #1e293b; margin-bottom: 25px; line-height: 1.7; }
                .info-card { background: #faf5ff; border: 2px solid #e9d5ff; border-radius: 8px; padding: 25px; margin-bottom: 25px; }
                .info-card h2 { color: #7c3aed; font-size: 20px; margin-bottom: 20px; font-weight: 600; }
                .features { background: #f8fafc; border-radius: 8px; padding: 20px; margin: 20px 0; }
                .features h3 { color: #1e293b; font-size: 16px; margin-bottom: 15px; font-weight: 600; }
                .features ul { list-style: none; padding: 0; }
                .features li { padding: 8px 0; border-bottom: 1px solid #e2e8f0; display: flex; align-items: center; }
                .features li:last-child { border-bottom: none; }
                .features li:before { content: '✓'; color: #10b981; font-weight: bold; margin-right: 10px; }
                .cta-button { display: inline-block; background: #8b5cf6; color: white; padding: 12px 24px; text-decoration: none; border-radius: 6px; font-weight: 600; margin: 20px 0; }
                .cta-button:hover { background: #7c3aed; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                                 <div class='header'>
                     <h1>Welcome to Migration.ng!</h1>
                     <p>Your journey starts here</p>
                 </div>
                <div class='content'>
                    <div class='welcome-text'>
                        Dear <strong>" . htmlspecialchars($userData['firstname']) . "</strong>,
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin-bottom: 20px; line-height: 1.7;'>
                        Welcome to Migration.ng! We're excited to have you on board. Your account has been successfully created and you're now ready to start your educational journey with us.
                    </p>
                    
                    <div class='info-card'>
                        <h2>🎉 Account Created Successfully</h2>
                        <p style='color: #7c3aed; margin-bottom: 0;'>You can now access all our services and start your application process.</p>
                    </div>
                    
                    <div class='features'>
                        <h3>What you can do now:</h3>
                        <ul>
                            <li>Submit university applications</li>
                            <li>Track your application status</li>
                            <li>Upload required documents</li>
                            <li>Receive updates and notifications</li>
                            <li>Access our support resources</li>
                    </ul>
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin: 25px 0; line-height: 1.7;'>
                        You can log in to your account using your email address and the password you chose during registration.
                    </p>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://migration.ng/portal/login.php' class='cta-button'>Login to Your Account</a>
                    </div>
                    
                                         <p style='font-size: 14px; color: #64748b; text-align: center; margin-top: 30px;'>
                         If you have any questions or need assistance, please don't hesitate to contact us at <a href='mailto:services@migration.ng'>services@migration.ng</a>
                     </p>
                </div>
                <div class='footer'>
                     <p>Best regards,<br><strong>The Migration.ng Team</strong></p>
                     <div style='text-align: center; margin: 20px 0;'>
                         <img src='https://migration.ng/assets/images/m..png' alt='Migration.ng' style='width: 100px; height: auto;'>
                     </div>
                     <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }
    
    /**
     * Build application status update email content
     * @param array $data
     * @return string
     */
    private static function buildApplicationStatusEmail($data) {
        // Determine status color and icon based on status
        $statusColors = [
            'Under Review' => ['bg' => '#fef3c7', 'border' => '#fbbf24', 'text' => '#92400e', 'icon' => '🔍'],
            'Approved' => ['bg' => '#d1fae5', 'border' => '#10b981', 'text' => '#065f46', 'icon' => '✅'],
            'Rejected' => ['bg' => '#fee2e2', 'border' => '#ef4444', 'text' => '#991b1b', 'icon' => '❌'],
            'Pending Documents' => ['bg' => '#dbeafe', 'border' => '#3b82f6', 'text' => '#1e40af', 'icon' => '📄'],
            'Interview Scheduled' => ['bg' => '#f3e8ff', 'border' => '#8b5cf6', 'text' => '#6b21a8', 'icon' => '📅'],
            'Accepted' => ['bg' => '#d1fae5', 'border' => '#10b981', 'text' => '#065f46', 'icon' => '🎉'],
            'Waitlisted' => ['bg' => '#fef3c7', 'border' => '#f59e0b', 'text' => '#92400e', 'icon' => '⏳']
        ];
        
        $statusInfo = $statusColors[$data['status']] ?? $statusColors['Under Review'];
        
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>Application Status Update - Migration.ng</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .welcome-text { font-size: 18px; color: #1e293b; margin-bottom: 25px; line-height: 1.7; }
                .status-card { background: " . $statusInfo['bg'] . "; border: 2px solid " . $statusInfo['border'] . "; border-radius: 8px; padding: 25px; margin-bottom: 25px; text-align: center; }
                .status-card h2 { color: " . $statusInfo['text'] . "; font-size: 24px; margin-bottom: 10px; font-weight: 600; }
                .status-icon { font-size: 48px; margin-bottom: 15px; }
                .status-card p { color: " . $statusInfo['text'] . "; font-size: 16px; margin: 0; }
                .app-details { background: #f8fafc; border-radius: 8px; padding: 20px; margin: 20px 0; }
                .app-details h3 { color: #1e293b; font-size: 16px; margin-bottom: 15px; font-weight: 600; }
                .app-details ul { list-style: none; padding: 0; }
                .app-details li { padding: 8px 0; border-bottom: 1px solid #e2e8f0; }
                .app-details li:last-child { border-bottom: none; }
                .app-details strong { color: #475569; }
                .cta-button { display: inline-block; background: #3b82f6; color: white; padding: 12px 24px; text-decoration: none; border-radius: 6px; font-weight: 600; margin: 20px 0; }
                .cta-button:hover { background: #2563eb; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                                 <div class='header'>
                     <h1>Application Status Update</h1>
                     <p>Your application has been updated</p>
                 </div>
                <div class='content'>
                    <div class='welcome-text'>
                        Dear <strong>" . htmlspecialchars($data['firstname']) . "</strong>,
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin-bottom: 20px; line-height: 1.7;'>
                        Your university application has been updated. Here's the latest status of your application.
                    </p>
                    
                    <div class='status-card'>
                        <div class='status-icon'>" . $statusInfo['icon'] . "</div>
                        <h2>" . htmlspecialchars($data['status']) . "</h2>
                        <p>Your application status has been updated</p>
                    </div>
                    
                    <div class='app-details'>
                        <h3>Application Summary</h3>
                        <ul>
                            <li><strong>Country:</strong> " . htmlspecialchars($data['country']) . "</li>
                            <li><strong>University:</strong> " . htmlspecialchars($data['university']) . "</li>
                            <li><strong>Degree Level:</strong> " . htmlspecialchars(html_entity_decode($data['degree'])) . "</li>
                            <li><strong>Program:</strong> " . htmlspecialchars($data['program']) . "</li>
                            <li><strong>Current Status:</strong> " . htmlspecialchars($data['status']) . "</li>
                            <li><strong>Update Date:</strong> " . date('F j, Y \a\t g:i A') . "</li>
                        </ul>
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin: 25px 0; line-height: 1.7;'>
                        We will contact you at <strong>" . htmlspecialchars($data['email']) . "</strong> with further instructions and updates regarding your application.
                    </p>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://migration.ng/portal/applications.php' class='cta-button'>View Application Details</a>
                    </div>
                    
                                         <p style='font-size: 14px; color: #64748b; text-align: center; margin-top: 30px;'>
                         If you have any questions about this update, please contact us at <a href='mailto:services@migration.ng'>services@migration.ng</a>
                     </p>
                 </div>
                 <div class='footer'>
                     <p>Best regards,<br><strong>The Migration.ng Team</strong></p>
                     <div style='text-align: center; margin: 20px 0;'>
                         <img src='https://migration.ng/assets/images/m.-preview.png' alt='Migration.ng' style='width: 100px; height: auto;'>
                     </div>
                     <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                 </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }
    
    /**
     * Send order submitted notification to admin
     * @param array $request
     * @param array $user
     * @return bool
     */
    public static function sendOrderSubmittedNotification($request, $user) {
        $to = "franciis.scott@gmail.com";
        $subject = "New Document Request Order Submitted - Migration.ng";
        
        $message = self::buildOrderSubmittedEmail($request, $user);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Build order submitted notification email
     * @param array $request
     * @param array $user
     * @return string
     */
    private static function buildOrderSubmittedEmail($request, $user) {
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>New Order Submitted - Migration.ng</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .welcome-text { font-size: 18px; color: #1e293b; margin-bottom: 25px; line-height: 1.7; }
                .order-card { background: #f0f9ff; border: 2px solid #0ea5e9; border-radius: 8px; padding: 25px; margin-bottom: 25px; text-align: center; }
                .order-card h2 { color: #0c4a6e; font-size: 24px; margin-bottom: 10px; font-weight: 600; }
                .order-icon { font-size: 48px; margin-bottom: 15px; color: #0ea5e9; }
                .order-card p { color: #0c4a6e; font-size: 16px; margin: 0; }
                .request-details { background: #f8fafc; border-radius: 8px; padding: 20px; margin: 20px 0; }
                .request-details h3 { color: #1e293b; font-size: 16px; margin-bottom: 15px; font-weight: 600; }
                .request-details ul { list-style: none; padding: 0; }
                .request-details li { padding: 8px 0; border-bottom: 1px solid #e2e8f0; }
                .request-details li:last-child { border-bottom: none; }
                .request-details strong { color: #475569; }
                .user-details { background: #f0fdf4; border-radius: 8px; padding: 20px; margin: 20px 0; }
                .user-details h3 { color: #1e293b; font-size: 16px; margin-bottom: 15px; font-weight: 600; }
                .user-details ul { list-style: none; padding: 0; }
                .user-details li { padding: 8px 0; border-bottom: 1px solid #e2e8f0; }
                .user-details li:last-child { border-bottom: none; }
                .user-details strong { color: #475569; }
                .cta-button { display: inline-block; background: #3b82f6; color: white; padding: 12px 24px; text-decoration: none; border-radius: 6px; font-weight: 600; margin: 20px 0; }
                .cta-button:hover { background: #2563eb; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>New Order Submitted</h1>
                    <p>A new document request order has been submitted</p>
                </div>
                <div class='content'>
                    <div class='welcome-text'>
                        <strong>New Document Request Order</strong>
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin-bottom: 20px; line-height: 1.7;'>
                        A new document request order has been submitted and is ready for review and invoicing.
                    </p>
                    
                    <div class='order-card'>
                        <div class='order-icon'>📋</div>
                        <h2>Order #" . htmlspecialchars($request['request_number']) . "</h2>
                        <p>Ready for review and invoicing</p>
                    </div>
                    
                    <div class='request-details'>
                        <h3>Request Details</h3>
                        <ul>
                            <li><strong>Request Number:</strong> " . htmlspecialchars($request['request_number']) . "</li>
                            <li><strong>Document Type:</strong> " . htmlspecialchars($request['document_type']) . "</li>
                            <li><strong>Destination Country:</strong> " . htmlspecialchars($request['destination_country']) . "</li>
                            <li><strong>Urgency Level:</strong> " . ucfirst($request['urgency_level']) . "</li>
                            <li><strong>Estimated Amount:</strong> ₦" . number_format($request['total_amount'], 2) . "</li>
                            <li><strong>Submission Date:</strong> " . date('F j, Y \a\t g:i A') . "</li>
                        </ul>
                    </div>
                    
                    <div class='user-details'>
                        <h3>Customer Information</h3>
                        <ul>
                            <li><strong>Name:</strong> " . htmlspecialchars($user['firstname'] . ' ' . $user['lastname']) . "</li>
                            <li><strong>Email:</strong> " . htmlspecialchars($user['email']) . "</li>
                            <li><strong>Phone:</strong> " . htmlspecialchars($user['phone'] ?? 'Not provided') . "</li>
                            <li><strong>User ID:</strong> " . $user['id'] . "</li>
                        </ul>
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin: 25px 0; line-height: 1.7;'>
                        Please review this order and send an invoice to the customer with the exact pricing.
                    </p>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://migration.ng/portal/admin.php' class='cta-button'>Review Order</a>
                    </div>
                    
                    <p style='font-size: 14px; color: #64748b; text-align: center; margin-top: 30px;'>
                        This is an automated notification. Please review the order details and take appropriate action.
                    </p>
                </div>
                <div class='footer'>
                    <p>Best regards,<br><strong>The Migration.ng System</strong></p>
                    <div style='text-align: center; margin: 20px 0;'>
                        <img src='https://migration.ng/assets/images/m.-preview.png' alt='Migration.ng' style='width: 100px; height: auto;'>
                    </div>
                    <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }

    public static function buildApplicationStatusUpdateEmail($application, $status, $note, $admin) {
        $logoUrl = "https://migration.ng/assets/images/m.-preview.png";
        
        $statusColors = [
            'Pending' => '#f59e0b',
            'In Progress' => '#3b82f6', 
            'Completed' => '#10b981',
            'Rejected' => '#ef4444'
        ];
        
        $statusColor = $statusColors[$status] ?? '#6b7280';
        
        $email = "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='utf-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1'>
            <title>Application Status Update - Migration.ng</title>
        </head>
        <body style='margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f8fafc;'>
            <div style='max-width: 600px; margin: 0 auto; background-color: #ffffff;'>
                <!-- Header -->
                <div style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 30px; text-align: center;'>
                    <h1 style='color: #ffffff; margin: 0; font-size: 28px; font-weight: 600;'>Application Status Update</h1>
                    <p style='color: #e2e8f0; margin: 10px 0 0 0; font-size: 16px;'>Your university application status has been updated</p>
                </div>
                
                <!-- Content -->
                <div style='padding: 40px 30px;'>
                    <div style='background-color: #f0f9ff; border-left: 4px solid #3b82f6; padding: 20px; margin-bottom: 30px; border-radius: 0 8px 8px 0;'>
                        <h2 style='color: #1e40af; margin: 0 0 10px 0; font-size: 20px;'>Application Details</h2>
                        <div style='color: #1e3a8a;'>
                            <p style='margin: 5px 0;'><strong>Application Number:</strong> {$application['application_number']}</p>
                            <p style='margin: 5px 0;'><strong>Degree:</strong> " . html_entity_decode($application['degree']) . "</p>
                            <p style='margin: 5px 0;'><strong>University:</strong> {$application['university']}</p>
                            <p style='margin: 5px 0;'><strong>Country:</strong> {$application['country']}</p>
                            <p style='margin: 5px 0;'><strong>Application Date:</strong> " . date('F j, Y', strtotime($application['created_at'])) . "</p>
                        </div>
                    </div>
                    
                    <div style='background-color: #fef3c7; border-left: 4px solid {$statusColor}; padding: 20px; margin-bottom: 30px; border-radius: 0 8px 8px 0;'>
                        <h3 style='color: #92400e; margin: 0 0 10px 0; font-size: 18px;'>Status Update</h3>
                        <div style='color: #78350f;'>
                            <p style='margin: 5px 0;'><strong>New Status:</strong> <span style='color: {$statusColor}; font-weight: bold;'>{$status}</span></p>
                            <p style='margin: 5px 0;'><strong>Updated By:</strong> {$admin['firstname']} {$admin['lastname']}</p>
                            <p style='margin: 5px 0;'><strong>Update Date:</strong> " . date('F j, Y \a\t g:i A') . "</p>
                        </div>
                    </div>";
        
        if (!empty($note)) {
            $email .= "
                    <div style='background-color: #fef2f2; border-left: 4px solid #ef4444; padding: 20px; margin-bottom: 30px; border-radius: 0 8px 8px 0;'>
                        <h3 style='color: #991b1b; margin: 0 0 10px 0; font-size: 18px;'>Admin Note</h3>
                        <div style='color: #7f1d1d;'>
                            <p style='margin: 0; font-style: italic;'>{$note}</p>
                        </div>
                    </div>";
        }
        
        $email .= "
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://migration.ng/portal/application-details.php?id={$application['id']}' style='display: inline-block; background-color: #3b82f6; color: #ffffff; padding: 12px 30px; text-decoration: none; border-radius: 6px; font-weight: 600;'>View Application Details</a>
                    </div>
                    
                    <div style='background-color: #f8fafc; padding: 20px; border-radius: 8px; margin-top: 30px;'>
                        <p style='margin: 0; color: #64748b; font-size: 14px; text-align: center;'>
                            If you have any questions about your application, please contact our support team at 
                            <a href='mailto:services@migration.ng' style='color: #3b82f6;'>services@migration.ng</a>
                        </p>
                    </div>
                </div>
                
                <!-- Footer -->
                <div style='background-color: #1f2937; padding: 30px; text-align: center;'>
                    <img src='{$logoUrl}' alt='Migration.ng' style='height: 40px; margin-bottom: 20px;'>
                    <p style='color: #9ca3af; margin: 0; font-size: 14px;'>
                        © " . date('Y') . " Migration.ng. All rights reserved.<br>
                        Your trusted partner for migration services.
                    </p>
                </div>
            </div>
        </body>
        </html>";
        
        return $email;
    }
    
    /**
     * Send document request update notification
     * @param array $documentRequest
     * @param array $admin
     * @return bool
     */
    public static function sendDocumentRequestUpdateNotification($documentRequest, $admin) {
        $to = $documentRequest['email'];
        $subject = "New Update on Document Request #{$documentRequest['request_number']} - Migration.ng";
        
        $message = self::buildDocumentRequestUpdateEmail($documentRequest, $admin);
        
        return self::sendEmail($to, $subject, $message);
    }
    
    /**
     * Build document request update email template
     * @param array $documentRequest
     * @param array $admin
     * @return string
     */
    private static function buildDocumentRequestUpdateEmail($documentRequest, $admin) {
        $logoUrl = "https://migration.ng/assets/images/m.-preview.png";
        
        $email = "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='utf-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1'>
            <title>Document Request Update - Migration.ng</title>
        </head>
        <body style='margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f8fafc;'>
            <div style='max-width: 600px; margin: 0 auto; background-color: #ffffff;'>
                <!-- Header -->
                <div style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 30px; text-align: center;'>
                    <h1 style='color: #ffffff; margin: 0; font-size: 28px; font-weight: 600;'>Document Request Update</h1>
                    <p style='color: #e2e8f0; margin: 10px 0 0 0; font-size: 16px;'>A new update has been added to your document request</p>
                </div>
                
                <!-- Content -->
                <div style='padding: 40px 30px;'>
                    <div style='background-color: #f0f9ff; border-left: 4px solid #3b82f6; padding: 20px; margin-bottom: 30px; border-radius: 0 8px 8px 0;'>
                        <h2 style='color: #1e40af; margin: 0 0 10px 0; font-size: 20px;'>Document Request Details</h2>
                        <div style='color: #1e3a8a;'>
                            <p style='margin: 5px 0;'><strong>Request Number:</strong> {$documentRequest['request_number']}</p>
                            <p style='margin: 5px 0;'><strong>Service Type:</strong> {$documentRequest['service_name']}</p>
                            <p style='margin: 5px 0;'><strong>Document Type:</strong> {$documentRequest['document_type']}</p>
                            <p style='margin: 5px 0;'><strong>Destination Country:</strong> {$documentRequest['destination_country']}</p>
                            <p style='margin: 5px 0;'><strong>Request Date:</strong> " . date('F j, Y', strtotime($documentRequest['created_at'])) . "</p>
                        </div>
                    </div>
                    
                    <div style='background-color: #fef3c7; border-left: 4px solid #f59e0b; padding: 20px; margin-bottom: 30px; border-radius: 0 8px 8px 0;'>
                        <h3 style='color: #92400e; margin: 0 0 10px 0; font-size: 18px;'>Update Notification</h3>
                        <div style='color: #78350f;'>
                            <p style='margin: 5px 0;'><strong>Update Date:</strong> " . date('F j, Y \a\t g:i A') . " GMT</p>
                            <p style='margin: 10px 0 0 0; font-weight: 600;'>🔒 <strong>IMPORTANT:</strong> You must log in to your portal to read this update.</p>
                            <p style='margin: 8px 0 0 0; font-style: italic; font-size: 14px;'>For your security and privacy, update details are only accessible through your secure portal account.</p>
                        </div>
                    </div>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://migration.ng/portal/document-request-details.php?id={$documentRequest['id']}' style='display: inline-block; background-color: #3b82f6; color: #ffffff; padding: 12px 30px; text-decoration: none; border-radius: 6px; font-weight: 600;'>View Update in Portal</a>
                    </div>
                    
                    <div style='background-color: #f8fafc; padding: 20px; border-radius: 8px; margin-top: 30px;'>
                        <p style='margin: 0; color: #64748b; font-size: 14px; text-align: center;'>
                            If you have any questions about your document request, please contact our support team at 
                            <a href='mailto:services@migration.ng' style='color: #3b82f6;'>services@migration.ng</a>
                        </p>
                    </div>
                </div>
                
                <!-- Footer -->
                <div style='background-color: #1f2937; padding: 30px; text-align: center;'>
                    <img src='{$logoUrl}' alt='Migration.ng' style='height: 40px; margin-bottom: 20px;'>
                    <p style='color: #9ca3af; margin: 0; font-size: 14px;'>
                        © " . date('Y') . " Migration.ng. All rights reserved.<br>
                        Your trusted partner for migration services.
                    </p>
                </div>
            </div>
        </body>
        </html>";
        
        return $email;
    }
    
    /**
     * Build email verification email content
     * @param array $userData
     * @param string $verificationToken
     * @return string
     */
    private static function buildEmailVerificationEmail($userData, $verificationToken) {
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>Verify Your Email - Migration.ng</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .welcome-text { font-size: 18px; color: #1e293b; margin-bottom: 25px; line-height: 1.7; }
                .info-card { background: #f0f9ff; border: 2px solid #0ea5e9; border-radius: 8px; padding: 25px; margin-bottom: 25px; }
                .info-card h2 { color: #0c4a6e; font-size: 20px; margin-bottom: 20px; font-weight: 600; }
                .verify-button { display: inline-block; background: #3b82f6; color: white; padding: 16px 32px; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 16px; margin: 25px 0; text-align: center; }
                .verify-button:hover { background: #2563eb; }
                .security-notice { background: #fef2f2; border: 2px solid #fecaca; border-radius: 8px; padding: 20px; margin: 25px 0; }
                .security-notice h3 { color: #dc2626; font-size: 16px; margin-bottom: 10px; font-weight: 600; }
                .security-notice p { color: #7f1d1d; font-size: 14px; margin: 0; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>Verify Your Email</h1>
                    <p>Complete your account registration</p>
                </div>
                <div class='content'>
                    <div class='welcome-text'>
                        Dear <strong>" . htmlspecialchars($userData['firstname']) . "</strong>,
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin-bottom: 20px; line-height: 1.7;'>
                        Thank you for registering with Migration.ng! To complete your account setup and start using our services, please verify your email address by clicking the button below.
                    </p>
                    
                    <div class='info-card'>
                        <h2>📧 Email Verification Required</h2>
                        <p style='color: #0c4a6e; margin-bottom: 0;'>Please verify your email address to activate your account and access all features.</p>
                    </div>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://migration.ng/portal/verify-email.php?token=" . htmlspecialchars($verificationToken) . "' class='verify-button'>Verify My Email</a>
                    </div>
                    
                    <div class='security-notice'>
                        <h3>⚠️ Security Notice</h3>
                        <p>• This verification link will expire in 24 hours for security reasons</p>
                        <p>• If you did not create this account, please ignore this email</p>
                        <p>• Never share this link with anyone</p>
                    </div>
                    
                    <p style='font-size: 14px; color: #64748b; text-align: center; margin-top: 30px;'>
                        If the button doesn't work, copy and paste this link into your browser:<br>
                        <a href='https://migration.ng/portal/verify-email.php?token=" . htmlspecialchars($verificationToken) . "'>https://migration.ng/portal/verify-email.php?token=" . htmlspecialchars($verificationToken) . "</a>
                    </p>
                </div>
                <div class='footer'>
                    <p>Best regards,<br><strong>The Migration.ng Team</strong></p>
                    <div style='text-align: center; margin: 20px 0;'>
                        <img src='https://migration.ng/assets/images/m..png' alt='Migration.ng' style='width: 100px; height: auto;'>
                    </div>
                    <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }
    
    /**
     * Build email verification reminder email content
     * @param array $userData
     * @param string $verificationToken
     * @return string
     */
    private static function buildEmailVerificationReminderEmail($userData, $verificationToken) {
        $html = "<!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <title>Reminder: Verify Your Email - Migration.ng</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
                .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                .header { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; padding: 40px 30px; text-align: center; }
                .logo { width: 120px; height: auto; margin-bottom: 20px; }
                .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
                .header p { font-size: 16px; opacity: 0.9; }
                .content { padding: 40px 30px; background: #ffffff; }
                .welcome-text { font-size: 18px; color: #1e293b; margin-bottom: 25px; line-height: 1.7; }
                .info-card { background: #fffbeb; border: 2px solid #fde68a; border-radius: 8px; padding: 25px; margin-bottom: 25px; }
                .info-card h2 { color: #92400e; font-size: 20px; margin-bottom: 20px; font-weight: 600; }
                .verify-button { display: inline-block; background: #f59e0b; color: white; padding: 16px 32px; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 16px; margin: 25px 0; text-align: center; }
                .verify-button:hover { background: #d97706; }
                .security-notice { background: #fef2f2; border: 2px solid #fecaca; border-radius: 8px; padding: 20px; margin: 25px 0; }
                .security-notice h3 { color: #dc2626; font-size: 16px; margin-bottom: 10px; font-weight: 600; }
                .security-notice p { color: #7f1d1d; font-size: 14px; margin: 0; }
                .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
                .footer a { color: #3b82f6; text-decoration: none; }
                .footer a:hover { text-decoration: underline; }
                @media (max-width: 600px) {
                    .container { margin: 10px; border-radius: 8px; }
                    .header { padding: 30px 20px; }
                    .content { padding: 30px 20px; }
                }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>Email Verification Reminder</h1>
                    <p>Complete your account activation</p>
                </div>
                <div class='content'>
                    <div class='welcome-text'>
                        Dear <strong>" . htmlspecialchars($userData['firstname']) . "</strong>,
                    </div>
                    
                    <p style='font-size: 16px; color: #374151; margin-bottom: 20px; line-height: 1.7;'>
                        We noticed that you haven't verified your email address yet. To access all features of your Migration.ng account, please verify your email address by clicking the button below.
                    </p>
                    
                    <div class='info-card'>
                        <h2>⏰ Verification Reminder</h2>
                        <p style='color: #92400e; margin-bottom: 0;'>Your account is waiting for email verification to be fully activated.</p>
                    </div>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://migration.ng/portal/verify-email.php?token=" . htmlspecialchars($verificationToken) . "' class='verify-button'>Verify My Email Now</a>
                    </div>
                    
                    <div class='security-notice'>
                        <h3>⚠️ Security Notice</h3>
                        <p>• This verification link will expire in 24 hours for security reasons</p>
                        <p>• If you did not create this account, please ignore this email</p>
                        <p>• Never share this link with anyone</p>
                    </div>
                    
                    <p style='font-size: 14px; color: #64748b; text-align: center; margin-top: 30px;'>
                        If the button doesn't work, copy and paste this link into your browser:<br>
                        <a href='https://migration.ng/portal/verify-email.php?token=" . htmlspecialchars($verificationToken) . "'>https://migration.ng/portal/verify-email.php?token=" . htmlspecialchars($verificationToken) . "</a>
                    </p>
                </div>
                <div class='footer'>
                    <p>Best regards,<br><strong>The Migration.ng Team</strong></p>
                    <div style='text-align: center; margin: 20px 0;'>
                        <img src='https://migration.ng/assets/images/m..png' alt='Migration.ng' style='width: 100px; height: auto;'>
                    </div>
                    <p style='margin-top: 20px; font-size: 12px;'>© " . date('Y') . " Migration.ng. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
        
        return $html;
    }
} 