<?php
/**
 * Email Verification Class
 * Handles email verification functionality for user registration
 */

class EmailVerification {
    
    /**
     * Generate a secure verification token
     * @return string
     */
    public static function generateVerificationToken() {
        return bin2hex(random_bytes(32));
    }
    
    /**
     * Create email verification for a user
     * @param int $userId
     * @param string $email
     * @return string|false Returns token on success, false on failure
     */
    public static function createVerification($userId, $email) {
        try {
            $pdo = getDatabaseConnection();
            
            // Generate verification token
            $token = self::generateVerificationToken();
            
            // Set expiration time (24 hours from now)
            $expires = date('Y-m-d H:i:s', strtotime('+24 hours'));
            
            // Update user with verification token
            $stmt = $pdo->prepare("UPDATE users SET email_verification_token = ?, email_verification_expires = ? WHERE id = ? AND email = ?");
            $result = $stmt->execute([$token, $expires, $userId, $email]);
            
            if ($result) {
                return $token;
            }
            
            return false;
        } catch (Exception $e) {
            error_log("Error creating email verification: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Verify email verification token
     * @param string $token
     * @return array|false Returns user data on success, false on failure
     */
    public static function verifyToken($token) {
        try {
            $pdo = getDatabaseConnection();
            
            // Find user with valid token
            $stmt = $pdo->prepare("SELECT * FROM users WHERE email_verification_token = ? AND email_verification_expires > NOW() AND email_verified = 0");
            $stmt->execute([$token]);
            $user = $stmt->fetch();
            
            if ($user) {
                // Mark email as verified and clear token
                $updateStmt = $pdo->prepare("UPDATE users SET email_verified = 1, email_verification_token = NULL, email_verification_expires = NULL WHERE id = ?");
                $updateStmt->execute([$user['id']]);
                
                return $user;
            }
            
            return false;
        } catch (Exception $e) {
            error_log("Error verifying email token: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Check if user's email is verified
     * @param int $userId
     * @return bool
     */
    public static function isEmailVerified($userId) {
        try {
            $pdo = getDatabaseConnection();
            
            $stmt = $pdo->prepare("SELECT email_verified FROM users WHERE id = ?");
            $stmt->execute([$userId]);
            $result = $stmt->fetch();
            
            return $result && $result['email_verified'] == 1;
        } catch (Exception $e) {
            error_log("Error checking email verification status: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get user by email verification token
     * @param string $token
     * @return array|false
     */
    public static function getUserByToken($token) {
        try {
            $pdo = getDatabaseConnection();
            
            $stmt = $pdo->prepare("SELECT * FROM users WHERE email_verification_token = ? AND email_verification_expires > NOW()");
            $stmt->execute([$token]);
            
            return $stmt->fetch();
        } catch (Exception $e) {
            error_log("Error getting user by token: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Resend verification email
     * @param string $email
     * @return bool
     */
    public static function resendVerification($email) {
        try {
            $pdo = getDatabaseConnection();
            
            // Get user by email
            $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? AND email_verified = 0");
            $stmt->execute([$email]);
            $user = $stmt->fetch();
            
            if (!$user) {
                return false;
            }
            
            // Generate new verification token
            $token = self::createVerification($user['id'], $email);
            
            if ($token) {
                // Send verification email
                return EmailService::sendEmailVerification($user, $token);
            }
            
            return false;
        } catch (Exception $e) {
            error_log("Error resending verification email: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Clean up expired verification tokens
     * @return int Number of cleaned up tokens
     */
    public static function cleanupExpiredTokens() {
        try {
            $pdo = getDatabaseConnection();
            
            $stmt = $pdo->prepare("UPDATE users SET email_verification_token = NULL, email_verification_expires = NULL WHERE email_verification_expires < NOW()");
            $stmt->execute();
            
            return $stmt->rowCount();
        } catch (Exception $e) {
            error_log("Error cleaning up expired tokens: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Get verification status message
     * @param string $token
     * @return string
     */
    public static function getVerificationStatusMessage($token) {
        if (empty($token)) {
            return "Invalid verification link.";
        }
        
        $user = self::getUserByToken($token);
        
        if (!$user) {
            return "This verification link has expired or is invalid. Please request a new verification email.";
        }
        
        if ($user['email_verified'] == 1) {
            return "Your email has already been verified. You can now log in to your account.";
        }
        
        return "Email verification successful! Your account has been activated.";
    }
}
?> 