<?php
/**
 * Error Handling and Logging System
 * Centralized error handling for the application
 */

class ErrorHandler {
    
    /**
     * Initialize error handling
     */
    public static function init() {
        // Set error reporting
        error_reporting(E_ALL);
        ini_set('display_errors', 0);
        ini_set('log_errors', 1);
        ini_set('error_log', __DIR__ . '/../logs/error.log');
        
        // Set custom error handler
        set_error_handler([self::class, 'handleError']);
        set_exception_handler([self::class, 'handleException']);
        register_shutdown_function([self::class, 'handleFatalError']);
    }
    
    /**
     * Handle PHP errors
     * @param int $errno
     * @param string $errstr
     * @param string $errfile
     * @param int $errline
     * @return bool
     */
    public static function handleError($errno, $errstr, $errfile, $errline) {
        $error = [
            'type' => 'PHP Error',
            'level' => $errno,
            'message' => $errstr,
            'file' => $errfile,
            'line' => $errline,
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        self::logError($error);
        
        // Don't display errors to users in production
        if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
            echo "<div style='background: #fee; border: 1px solid #fcc; padding: 10px; margin: 10px;'>";
            echo "<strong>Error:</strong> " . htmlspecialchars($errstr) . "<br>";
            echo "<strong>File:</strong> " . htmlspecialchars($errfile) . ":" . $errline;
            echo "</div>";
        }
        
        return true;
    }
    
    /**
     * Handle exceptions
     * @param Exception $exception
     */
    public static function handleException($exception) {
        $error = [
            'type' => 'Exception',
            'class' => get_class($exception),
            'message' => $exception->getMessage(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString(),
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        self::logError($error);
        self::displayError($exception->getMessage());
    }
    
    /**
     * Handle fatal errors
     */
    public static function handleFatalError() {
        $error = error_get_last();
        if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            $errorData = [
                'type' => 'Fatal Error',
                'level' => $error['type'],
                'message' => $error['message'],
                'file' => $error['file'],
                'line' => $error['line'],
                'timestamp' => date('Y-m-d H:i:s')
            ];
            
            self::logError($errorData);
            self::displayError('A fatal error occurred. Please try again later.');
        }
    }
    
    /**
     * Log error to file
     * @param array $error
     */
    private static function logError($error) {
        $logDir = __DIR__ . '/../logs';
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $logFile = $logDir . '/error.log';
        $logEntry = json_encode($error) . "\n";
        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Display user-friendly error message
     * @param string $message
     */
    public static function displayError($message) {
        if (headers_sent()) {
            echo "<div style='background: #fee; border: 1px solid #fcc; padding: 15px; margin: 20px; border-radius: 5px;'>";
            echo "<h3 style='color: #c33; margin: 0 0 10px 0;'>Error</h3>";
            echo "<p style='margin: 0;'>" . htmlspecialchars($message) . "</p>";
            echo "</div>";
        } else {
            http_response_code(500);
            include __DIR__ . '/../templates/error.php';
        }
    }
    
    /**
     * Display validation error
     * @param string $message
     */
    public static function displayValidationError($message) {
        echo "<div class='bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-4' role='alert'>";
        echo "<span class='block sm:inline'>" . htmlspecialchars($message) . "</span>";
        echo "</div>";
    }
    
    /**
     * Display success message
     * @param string $message
     */
    public static function displaySuccess($message) {
        echo "<div class='bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative mb-4' role='alert'>";
        echo "<span class='block sm:inline'>" . htmlspecialchars($message) . "</span>";
        echo "</div>";
    }
} 