<?php
/**
 * User Authentication and Authorization Class
 * Handles user login, role checking, and permission management
 */

class UserAuth {
    
    /**
     * Check if user is logged in
     * @return bool
     */
    public static function isLoggedIn() {
        return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
    
    /**
     * Get current user ID
     * @return int|null
     */
    public static function getCurrentUserId() {
        return $_SESSION['user_id'] ?? null;
    }
    
    /**
     * Get current user data
     * @return array|null
     */
    public static function getCurrentUser() {
        if (!self::isLoggedIn()) {
            return null;
        }
        
        try {
            $pdo = getDatabaseConnection();
            $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND is_active = 1 AND email_verified = 1");
            $stmt->execute([self::getCurrentUserId()]);
            return $stmt->fetch();
        } catch (Exception $e) {
            error_log("Error getting current user: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Check if current user has admin role
     * @return bool
     */
    public static function isAdmin() {
        $user = self::getCurrentUser();
        return $user && $user['role'] === 'admin';
    }
    
    /**
     * Check if current user has moderator role
     * @return bool
     */
    public static function isModerator() {
        $user = self::getCurrentUser();
        return $user && ($user['role'] === 'moderator' || $user['role'] === 'admin');
    }
    
    /**
     * Check if current user has specific role
     * @param string $role
     * @return bool
     */
    public static function hasRole($role) {
        $user = self::getCurrentUser();
        return $user && $user['role'] === $role;
    }
    
    /**
     * Require admin access - redirect if not admin
     */
    public static function requireAdmin() {
        if (!self::isAdmin()) {
            header('Location: login.php?error=access_denied');
            exit();
        }
    }
    
    /**
     * Require moderator access - redirect if not moderator or admin
     */
    public static function requireModerator() {
        if (!self::isModerator()) {
            header('Location: login.php?error=access_denied');
            exit();
        }
    }
    
    /**
     * Require login - redirect if not logged in
     */
    public static function requireLogin() {
        if (!self::isLoggedIn()) {
            header('Location: login.php');
            exit();
        }
    }
    
    /**
     * Get user role display name
     * @param string $role
     * @return string
     */
    public static function getRoleDisplayName($role) {
        switch ($role) {
            case 'admin':
                return 'Administrator';
            case 'moderator':
                return 'Moderator';
            case 'user':
                return 'User';
            default:
                return 'Unknown';
        }
    }
    
    /**
     * Get all available roles
     * @return array
     */
    public static function getAvailableRoles() {
        return [
            'user' => 'User',
            'moderator' => 'Moderator',
            'admin' => 'Administrator'
        ];
    }
}
?> 