<?php
/**
 * Input Validation and Sanitization Class
 * Handles all form validation and data sanitization
 */

class Validator {
    
    /**
     * Validate and sanitize email address
     * @param string $email
     * @return string|false
     */
    public static function validateEmail($email) {
        $email = filter_var(trim($email), FILTER_SANITIZE_EMAIL);
        return filter_var($email, FILTER_VALIDATE_EMAIL) ? $email : false;
    }
    
    /**
     * Validate and sanitize text input
     * @param string $text
     * @param int $maxLength
     * @return string|false
     */
    public static function validateText($text, $maxLength = 255) {
        $text = trim($text);
        if (empty($text) || strlen($text) > $maxLength) {
            return false;
        }
        return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
    }
    
    /**
     * Validate country code
     * @param string $countryCode
     * @return string|false
     */
    public static function validateCountry($countryCode) {
        $countryCode = strtoupper(trim($countryCode));
        // Add validation against allowed countries if needed
        return preg_match('/^[A-Z]{2}$/', $countryCode) ? $countryCode : false;
    }
    
    /**
     * Validate file upload
     * @param array $file
     * @param array $allowedTypes
     * @param int $maxSize
     * @return array|false
     */
    public static function validateFile($file, $allowedTypes = ['jpg', 'jpeg', 'png', 'pdf', 'doc', 'docx'], $maxSize = 5242880) {
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            return false;
        }
        
        // Check file size
        if ($file['size'] > $maxSize) {
            return false;
        }
        
        // Check file type
        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($fileExtension, $allowedTypes)) {
            return false;
        }
        
        // Additional security check - verify file content
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        $allowedMimes = [
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'pdf' => 'application/pdf',
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ];
        
        if (!isset($allowedMimes[$fileExtension]) || $allowedMimes[$fileExtension] !== $mimeType) {
            return false;
        }
        
        return [
            'name' => $file['name'],
            'tmp_name' => $file['tmp_name'],
            'size' => $file['size'],
            'type' => $mimeType,
            'extension' => $fileExtension
        ];
    }
    
    /**
     * Generate safe filename
     * @param string $originalName
     * @return string
     */
    public static function generateSafeFilename($originalName) {
        $extension = pathinfo($originalName, PATHINFO_EXTENSION);
        $safeName = preg_replace('/[^a-zA-Z0-9_-]/', '_', pathinfo($originalName, PATHINFO_FILENAME));
        return $safeName . '_' . time() . '.' . $extension;
    }
    
    /**
     * Validate CSRF token
     * @param string $token
     * @return bool
     */
    public static function validateCSRFToken($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
    
    /**
     * Generate CSRF token
     * @return string
     */
    public static function generateCSRFToken() {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }
} 