<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || !UserAuth::isAdmin()) {
    header('Location: login.php');
    exit();
}

// Check if this is a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: admin-applications.php');
    exit();
}

// Validate required fields
$document_id = $_POST['document_id'] ?? '';
$update_text = $_POST['update_text'] ?? '';

if (empty($document_id) || empty($update_text)) {
    $_SESSION['error'] = "Document ID and update text are required.";
    header('Location: admin-applications.php');
    exit();
}

// Validate update text
$update_text = Validator::validateText($update_text, 1000);
if (!$update_text) {
    $_SESSION['error'] = "Please enter a valid update message.";
    header('Location: admin-applications.php');
    exit();
}

try {
    $pdo = getDatabaseConnection();
    
    // Get document request details with user information
    $stmt = $pdo->prepare("
        SELECT 
            dr.*,
            u.firstname,
            u.lastname,
            u.email,
            dst.name as service_name
        FROM document_requests dr
        LEFT JOIN users u ON dr.user_id = u.id
        LEFT JOIN document_service_types dst ON dr.service_type_id = dst.id
        WHERE dr.id = ?
    ");
    $stmt->execute([$document_id]);
    $documentRequest = $stmt->fetch();
    
    if (!$documentRequest) {
        $_SESSION['error'] = "Document request not found.";
        header('Location: admin-applications.php');
        exit();
    }
    
    // Get admin information
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $admin = $stmt->fetch();
    
    // Add update to timeline
    $stmt = $pdo->prepare("
        INSERT INTO document_request_updates 
        (request_id, user_id, update_text, update_type, created_at) 
        VALUES (?, ?, ?, 'admin', NOW())
    ");
    $stmt->execute([$document_id, $_SESSION['user_id'], $update_text]);
    

    
    // Send email notification to user about the update
    try {
        $emailResult = EmailService::sendDocumentRequestUpdateNotification($documentRequest, $admin);
        
        $_SESSION['success'] = "Update added successfully and notification sent to user.";
    } catch (Exception $e) {
        error_log("Failed to send update notification email: " . $e->getMessage());
        $_SESSION['success'] = "Update added successfully, but email notification failed.";
    }
    
} catch (Exception $e) {
    error_log("Add document update error: " . $e->getMessage());
    $_SESSION['error'] = "An error occurred while adding the update.";
}

// Redirect back to admin applications page
header('Location: admin-applications.php');
exit();
?> 