<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || !UserAuth::isAdmin()) {
    header('Location: login.php');
    exit();
}

// Get admin data and applications
try {
    $pdo = getDatabaseConnection();
    
    // Get admin data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $admin = $stmt->fetch();
    
    if (!$admin) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    // Get filter parameters
    $status_filter = $_GET['status'] ?? 'all';
    $search = $_GET['search'] ?? '';
    $type_filter = $_GET['type'] ?? 'all'; // all, applications, documents
    
    // Build the WHERE clause for applications
    $appWhereConditions = [];
    $appParams = [];
    
    if ($status_filter !== 'all') {
        $appWhereConditions[] = "pa.status = ?";
        $appParams[] = $status_filter;
    }
    
    if (!empty($search)) {
        $appWhereConditions[] = "(pa.degree LIKE ? OR pa.university LIKE ? OR pa.country LIKE ? OR u.firstname LIKE ? OR u.lastname LIKE ? OR u.email LIKE ?)";
        $searchParam = "%$search%";
        $appParams = array_merge($appParams, [$searchParam, $searchParam, $searchParam, $searchParam, $searchParam, $searchParam]);
    }
    
    $appWhereClause = !empty($appWhereConditions) ? "WHERE " . implode(" AND ", $appWhereConditions) : "";
    
    // Build the WHERE clause for document requests
    $docWhereConditions = [];
    $docParams = [];
    
    if ($status_filter !== 'all') {
        $docWhereConditions[] = "dr.status = ?";
        $docParams[] = $status_filter;
    }
    
    if (!empty($search)) {
        $docWhereConditions[] = "(dr.document_type LIKE ? OR dr.destination_country LIKE ? OR u.firstname LIKE ? OR u.lastname LIKE ? OR u.email LIKE ?)";
        $searchParam = "%$search%";
        $docParams = array_merge($docParams, [$searchParam, $searchParam, $searchParam, $searchParam, $searchParam]);
    }
    
    $docWhereClause = !empty($docWhereConditions) ? "WHERE " . implode(" AND ", $docWhereConditions) : "";
    
    // Get applications with user information
    $applications = [];
    if ($type_filter === 'all' || $type_filter === 'applications') {
        $stmt = $pdo->prepare("
            SELECT 
                pa.*,
                u.firstname,
                u.lastname,
                u.email,
                u.phone,
                'application' as type
            FROM portal_applications pa
            LEFT JOIN users u ON pa.user_id = u.id
            $appWhereClause
            ORDER BY pa.created_at DESC
        ");
        $stmt->execute($appParams);
        $applications = $stmt->fetchAll();
    }
    
    // Get document requests with user information
    $documentRequests = [];
    if ($type_filter === 'all' || $type_filter === 'documents') {
        $stmt = $pdo->prepare("
            SELECT 
                dr.*,
                u.firstname,
                u.lastname,
                u.email,
                u.phone,
                dst.name as service_name,
                'document' as type
            FROM document_requests dr
            LEFT JOIN users u ON dr.user_id = u.id
            LEFT JOIN document_service_types dst ON dr.service_type_id = dst.id
            $docWhereClause
            ORDER BY dr.created_at DESC
        ");
        $stmt->execute($docParams);
        $documentRequests = $stmt->fetchAll();
    }
    
    // Combine and sort all items
    $allItems = array_merge($applications, $documentRequests);
    usort($allItems, function($a, $b) {
        return strtotime($b['created_at']) - strtotime($a['created_at']);
    });
    
    // Get statistics for applications
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
            SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
        FROM portal_applications
    ");
    $stmt->execute();
    $appStats = $stmt->fetch();
    
    // Get statistics for document requests
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'submitted' THEN 1 ELSE 0 END) as submitted,
            SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
            SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled
        FROM document_requests
    ");
    $stmt->execute();
    $docStats = $stmt->fetch();
    
    // Combined statistics
    $stats = [
        'total' => ($appStats['total'] ?? 0) + ($docStats['total'] ?? 0),
        'pending' => ($appStats['pending'] ?? 0) + ($docStats['pending'] ?? 0),
        'submitted' => ($docStats['submitted'] ?? 0),
        'in_progress' => ($appStats['in_progress'] ?? 0) + ($docStats['in_progress'] ?? 0),
        'completed' => ($appStats['completed'] ?? 0) + ($docStats['completed'] ?? 0),
        'rejected' => ($appStats['rejected'] ?? 0),
        'cancelled' => ($docStats['cancelled'] ?? 0)
    ];
    
} catch (Exception $e) {
    error_log("Admin applications error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Get status color
function getStatusColor($status) {
    switch ($status) {
        case 'pending': return 'bg-yellow-100 text-yellow-800';
        case 'submitted': return 'bg-blue-100 text-blue-800';
        case 'in_progress': return 'bg-orange-100 text-orange-800';
        case 'completed': return 'bg-green-100 text-green-800';
        case 'rejected': return 'bg-red-100 text-red-800';
        case 'cancelled': return 'bg-gray-100 text-gray-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}

// Get status text
function getStatusText($status) {
    switch ($status) {
        case 'pending': return 'Pending';
        case 'submitted': return 'Submitted';
        case 'in_progress': return 'In Progress';
        case 'completed': return 'Completed';
        case 'rejected': return 'Rejected';
        case 'cancelled': return 'Cancelled';
        default: return ucfirst($status);
    }
}

// Convert URLs to clickable links
function makeLinksClickable($text) {
    // Pattern to match URLs
    $urlPattern = '/(https?:\/\/[^\s]+)/i';
    
    // Replace URLs with clickable links
    $text = preg_replace($urlPattern, '<a href="$1" target="_blank" class="text-blue-600 hover:text-blue-800 underline">$1</a>', $text);
    
    // Also handle URLs without http/https
    $urlPattern2 = '/(www\.[^\s]+)/i';
    $text = preg_replace($urlPattern2, '<a href="https://$1" target="_blank" class="text-blue-600 hover:text-blue-800 underline">$1</a>', $text);
    
    return $text;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin - Applications & Document Requests Management - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="admin.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Admin Panel</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="admin.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-red-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($admin['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($admin['firstname'] . ' ' . $admin['lastname']); ?></p>
                            <p class="text-xs text-gray-500">Admin</p>
                        </div>
                    </div>
                    <a href="logout.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-sign-out-alt"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Applications & Document Requests Management</h1>
                    <p class="text-gray-600 mt-2">Manage all user applications and document requests, update pricing, and track progress</p>
                </div>
                <a href="admin.php" class="text-gray-600 hover:text-gray-800">
                    <i class="fas fa-arrow-left mr-1"></i>
                    Back to Admin Panel
                </a>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (isset($error)): ?>
            <?php ErrorHandler::displayValidationError($error); ?>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-7 gap-6 mb-8">
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-blue-100 rounded-lg">
                        <i class="fas fa-file-alt text-blue-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Total Requests</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['total'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-yellow-100 rounded-lg">
                        <i class="fas fa-clock text-yellow-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Pending</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['pending'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-blue-100 rounded-lg">
                        <i class="fas fa-paper-plane text-blue-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Submitted</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['submitted'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-orange-100 rounded-lg">
                        <i class="fas fa-spinner text-orange-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">In Progress</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['in_progress'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-green-100 rounded-lg">
                        <i class="fas fa-check-circle text-green-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Completed</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['completed'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-red-100 rounded-lg">
                        <i class="fas fa-times-circle text-red-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Rejected</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['rejected'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-gray-100 rounded-lg">
                        <i class="fas fa-ban text-gray-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Cancelled</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['cancelled'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters and Search -->
        <div class="bg-white rounded-lg shadow mb-6">
            <div class="px-6 py-4">
                <form method="GET" class="flex flex-col md:flex-row gap-4">
                    <div class="flex-1">
                        <label for="search" class="block text-sm font-medium text-gray-700 mb-1">Search</label>
                        <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Search by degree, university, country, document type, or user name..." 
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div class="md:w-48">
                        <label for="type" class="block text-sm font-medium text-gray-700 mb-1">Type</label>
                        <select name="type" id="type" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="all" <?php echo $type_filter === 'all' ? 'selected' : ''; ?>>All Types</option>
                            <option value="applications" <?php echo $type_filter === 'applications' ? 'selected' : ''; ?>>University Applications</option>
                            <option value="documents" <?php echo $type_filter === 'documents' ? 'selected' : ''; ?>>Document Requests</option>
                        </select>
                    </div>
                    <div class="md:w-48">
                        <label for="status" class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                        <select name="status" id="status" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="all" <?php echo $status_filter === 'all' ? 'selected' : ''; ?>>All Status</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="submitted" <?php echo $status_filter === 'submitted' ? 'selected' : ''; ?>>Submitted</option>
                            <option value="in_progress" <?php echo $status_filter === 'in_progress' ? 'selected' : ''; ?>>In Progress</option>
                            <option value="completed" <?php echo $status_filter === 'completed' ? 'selected' : ''; ?>>Completed</option>
                            <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                            <option value="cancelled" <?php echo $status_filter === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                        </select>
                    </div>
                    <div class="flex items-end">
                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <i class="fas fa-search mr-2"></i>Filter
                        </button>
                        <?php if (!empty($search) || $status_filter !== 'all' || $type_filter !== 'all'): ?>
                            <a href="admin-applications.php" class="ml-2 px-4 py-2 bg-gray-600 text-white rounded-md hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-gray-500">
                                <i class="fas fa-times mr-2"></i>Clear
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
        </div>

        <!-- Requests List -->
        <div class="bg-white rounded-lg shadow">
            <div class="px-6 py-4 border-b border-gray-200">
                <h2 class="text-lg font-medium text-gray-900">
                    All Requests (<?php echo count($allItems); ?> found)
                </h2>
            </div>
            
            <?php if (empty($allItems)): ?>
                <div class="px-6 py-12 text-center">
                    <div class="mx-auto w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4">
                        <i class="fas fa-file-alt text-gray-400 text-2xl"></i>
                    </div>
                    <h3 class="text-lg font-medium text-gray-900 mb-2">No requests found</h3>
                    <p class="text-gray-500 mb-6">
                        <?php if (!empty($search) || $status_filter !== 'all' || $type_filter !== 'all'): ?>
                            No requests match your current filters. Try adjusting your search criteria.
                        <?php else: ?>
                            There are no requests in the system yet.
                        <?php endif; ?>
                    </p>
                </div>
            <?php else: ?>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Request Details</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Destination</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($allItems as $item): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="flex-shrink-0 h-10 w-10">
                                                <?php if ($item['type'] === 'application'): ?>
                                                    <div class="h-10 w-10 rounded-full bg-blue-100 flex items-center justify-center">
                                                        <i class="fas fa-graduation-cap text-blue-600"></i>
                                                    </div>
                                                <?php else: ?>
                                                    <div class="h-10 w-10 rounded-full bg-purple-100 flex items-center justify-center">
                                                        <i class="fas fa-stamp text-purple-600"></i>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                            <div class="ml-4">
                                                <div class="text-sm font-medium text-gray-900">
                                                    <?php echo $item['type'] === 'application' ? 'University Application' : 'Document Request'; ?>
                                                </div>
                                                <div class="text-sm text-gray-500">
                                                    #<?php echo htmlspecialchars($item['type'] === 'application' ? $item['application_number'] : $item['request_number']); ?>
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-900">
                                            <?php if ($item['type'] === 'application'): ?>
                                                <strong><?php echo htmlspecialchars($item['degree']); ?></strong><br>
                                                <span class="text-gray-500"><?php echo htmlspecialchars($item['university']); ?></span>
                                            <?php else: ?>
                                                <strong><?php echo htmlspecialchars($item['document_type']); ?></strong><br>
                                                <span class="text-gray-500"><?php echo htmlspecialchars($item['service_name']); ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-900">
                                            <strong><?php echo htmlspecialchars($item['firstname'] . ' ' . $item['lastname']); ?></strong>
                                        </div>
                                        <div class="text-sm text-gray-500">
                                            <?php echo htmlspecialchars($item['email']); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-900">
                                            <?php echo htmlspecialchars($item['type'] === 'application' ? $item['country'] : $item['destination_country']); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo getStatusColor($item['status']); ?>">
                                            <?php echo getStatusText($item['status']); ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo date('M j, Y', strtotime($item['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <?php if ($item['type'] === 'application'): ?>
                                            <a href="application-details.php?id=<?php echo $item['id']; ?>" class="text-blue-600 hover:text-blue-900 mr-3">View</a>
                                            <button onclick="openUpdateModal(<?php echo $item['id']; ?>, '<?php echo htmlspecialchars($item['status']); ?>', '<?php echo htmlspecialchars($item['firstname'] . ' ' . $item['lastname']); ?>')" class="text-green-600 hover:text-green-900">Update</button>
                                        <?php else: ?>
                                            <a href="document-request-details.php?id=<?php echo $item['id']; ?>" class="text-blue-600 hover:text-blue-900 mr-3">View</a>
                                            <?php if ($item['status'] === 'submitted'): ?>
                                                <button onclick="openDocumentUpdateModal(<?php echo $item['id']; ?>, '<?php echo htmlspecialchars($item['status']); ?>', '<?php echo htmlspecialchars($item['firstname'] . ' ' . $item['lastname']); ?>')" class="text-green-600 hover:text-green-900 mr-2">Update Price</button>
                                            <?php else: ?>
                                                <button onclick="openDocumentUpdateModal(<?php echo $item['id']; ?>, '<?php echo htmlspecialchars($item['status']); ?>', '<?php echo htmlspecialchars($item['firstname'] . ' ' . $item['lastname']); ?>')" class="text-green-600 hover:text-green-900 mr-2">Update</button>
                                            <?php endif; ?>
                                            <button onclick="openAddUpdateModal(<?php echo $item['id']; ?>)" class="text-purple-600 hover:text-purple-900">Add Update</button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Application Status Update Modal -->
    <div id="statusModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 class="text-lg font-medium text-gray-900">Update Application Status</h3>
                </div>
                <form id="statusForm" method="POST" action="update-application-status.php">
                    <div class="px-6 py-4">
                        <input type="hidden" id="applicationId" name="application_id">
                        <div class="mb-4">
                            <label for="newStatus" class="block text-sm font-medium text-gray-700 mb-2">New Status</label>
                            <select name="status" id="newStatus" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="pending">Pending</option>
                                <option value="in_progress">In Progress</option>
                                <option value="completed">Completed</option>
                                <option value="rejected">Rejected</option>
                            </select>
                        </div>
                        <div class="mb-4">
                            <label for="statusNote" class="block text-sm font-medium text-gray-700 mb-2">Note (Optional)</label>
                            <textarea name="note" id="statusNote" rows="3" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Add a note about this status update..."></textarea>
                        </div>
                    </div>
                    <div class="px-6 py-4 bg-gray-50 flex justify-end space-x-3">
                        <button type="button" onclick="closeStatusModal()" class="px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50">
                            Cancel
                        </button>
                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                            Update Status
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

         <!-- Document Request Update Modal -->
     <div id="documentModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
         <div class="flex items-center justify-center min-h-screen p-4">
             <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
                 <div class="px-6 py-4 border-b border-gray-200">
                     <h3 class="text-lg font-medium text-gray-900">Update Document Request</h3>
                 </div>
                 <form id="documentForm" method="POST" action="update-document-request.php">
                     <div class="px-6 py-4">
                         <input type="hidden" id="documentId" name="document_id">
                         <div class="mb-4">
                             <label for="documentStatus" class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                             <select name="status" id="documentStatus" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                 <option value="pending">Pending</option>
                                 <option value="submitted">Submitted</option>
                                 <option value="in_progress">In Progress</option>
                                 <option value="completed">Completed</option>
                                 <option value="cancelled">Cancelled</option>
                             </select>
                         </div>
                         <div class="mb-4">
                             <label for="documentPrice" class="block text-sm font-medium text-gray-700 mb-2">Final Price (₦)</label>
                             <input type="number" name="final_price" id="documentPrice" step="0.01" min="0" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Enter final price">
                         </div>
                         <div class="mb-4">
                             <label for="documentNote" class="block text-sm font-medium text-gray-700 mb-2">Note (Optional)</label>
                             <textarea name="note" id="documentNote" rows="3" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Add a note about this update... You can include payment links like: https://paystack.com/pay/your-payment-link"></textarea>
                         </div>
                     </div>
                     <div class="px-6 py-4 bg-gray-50 flex justify-end space-x-3">
                         <button type="button" onclick="closeDocumentModal()" class="px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50">
                             Cancel
                         </button>
                         <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                             Update Request
                         </button>
                     </div>
                 </form>
             </div>
         </div>
     </div>

     <!-- Admin Add Update Modal -->
     <div id="addUpdateModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
         <div class="flex items-center justify-center min-h-screen p-4">
             <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
                 <div class="px-6 py-4 border-b border-gray-200">
                     <h3 class="text-lg font-medium text-gray-900">Add Update to Document Request</h3>
                 </div>
                 <form id="addUpdateForm" method="POST" action="add-document-update.php">
                     <div class="px-6 py-4">
                         <input type="hidden" id="updateDocumentId" name="document_id">
                         <div class="mb-4">
                             <label for="updateText" class="block text-sm font-medium text-gray-700 mb-2">Update Message</label>
                             <textarea name="update_text" id="updateText" rows="4" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Add an update message... You can include payment links like: https://paystack.com/pay/your-payment-link"></textarea>
                             <p class="text-xs text-gray-500 mt-1">URLs will automatically become clickable links</p>
                         </div>
                     </div>
                     <div class="px-6 py-4 bg-gray-50 flex justify-end space-x-3">
                         <button type="button" onclick="closeAddUpdateModal()" class="px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50">
                             Cancel
                         </button>
                         <button type="submit" class="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700">
                             Add Update
                         </button>
                     </div>
                 </form>
             </div>
         </div>
     </div>

    <script>
        function openUpdateModal(applicationId, currentStatus, userName) {
            document.getElementById('applicationId').value = applicationId;
            document.getElementById('newStatus').value = currentStatus;
            document.getElementById('statusModal').classList.remove('hidden');
        }

        function openDocumentUpdateModal(documentId, currentStatus, userName) {
            document.getElementById('documentId').value = documentId;
            document.getElementById('documentStatus').value = currentStatus;
            document.getElementById('documentModal').classList.remove('hidden');
        }

        function openAddUpdateModal(documentId) {
            document.getElementById('updateDocumentId').value = documentId;
            document.getElementById('addUpdateModal').classList.remove('hidden');
        }

        function closeStatusModal() {
            document.getElementById('statusModal').classList.add('hidden');
        }

        function closeDocumentModal() {
            document.getElementById('documentModal').classList.add('hidden');
        }

        function closeAddUpdateModal() {
            document.getElementById('addUpdateModal').classList.add('hidden');
        }

        // Close modals when clicking outside
        document.getElementById('statusModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeStatusModal();
            }
        });

        document.getElementById('documentModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeDocumentModal();
            }
        });

        document.getElementById('addUpdateModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeAddUpdateModal();
            }
        });
    </script>
</body>
</html> 