<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Require admin access
UserAuth::requireAdmin();

// Get current user
$currentUser = UserAuth::getCurrentUser();

// Initialize variables
$errors = [];
$success = false;
$successMessage = '';

// Handle status updates
if (isset($_POST['update_status'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        $applicationId = (int)($_POST['application_id'] ?? 0);
        $newStatus = Validator::validateText($_POST['new_status'] ?? '', 20);
        $statusReason = Validator::validateText($_POST['status_reason'] ?? '', 500);
        
        if (!$applicationId || !$newStatus) {
            $errors[] = "Invalid application or status";
        } else {
            $pdo = getDatabaseConnection();
            
            // Update application status
            $stmt = $pdo->prepare("UPDATE portal_applications SET status = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$newStatus, $applicationId]);
            
            // Add status change to history
            $stmt = $pdo->prepare("INSERT INTO application_status_history (application_id, old_status, new_status, changed_by, change_reason, changed_at) VALUES (?, ?, ?, ?, ?, NOW())");
            $stmt->execute([$applicationId, 'pending', $newStatus, $currentUser['id'], $statusReason]);
            
            // Add update to application_updates table
            $updateText = "Status changed from pending to " . $newStatus;
            if ($statusReason) {
                $updateText .= ": " . $statusReason;
            }
            $stmt = $pdo->prepare("INSERT INTO application_updates (application_id, user_id, update_text, update_type, created_at) VALUES (?, ?, ?, 'admin', NOW())");
            $stmt->execute([$applicationId, $currentUser['id'], $updateText]);
            
            $success = true;
            $successMessage = "Application status updated successfully!";
        }
    } catch (Exception $e) {
        error_log("Status update error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Handle bulk actions
if (isset($_POST['bulk_action'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        $selectedApplications = $_POST['selected_applications'] ?? [];
        $bulkAction = $_POST['bulk_action_type'] ?? '';
        
        if (empty($selectedApplications)) {
            $errors[] = "Please select at least one application";
        } elseif (!in_array($bulkAction, ['approve', 'reject', 'in_progress'])) {
            $errors[] = "Invalid bulk action";
        } else {
            $pdo = getDatabaseConnection();
            $updatedCount = 0;
            
            foreach ($selectedApplications as $appId) {
                $appId = (int)$appId;
                if ($appId > 0) {
                    // Update application status
                    $stmt = $pdo->prepare("UPDATE portal_applications SET status = ?, updated_at = NOW() WHERE id = ? AND status = 'pending'");
                    $stmt->execute([$bulkAction, $appId]);
                    
                    if ($stmt->rowCount() > 0) {
                        // Add status change to history
                        $stmt = $pdo->prepare("INSERT INTO application_status_history (application_id, old_status, new_status, changed_by, change_reason, changed_at) VALUES (?, ?, ?, ?, ?, NOW())");
                        $stmt->execute([$appId, 'pending', $bulkAction, $currentUser['id'], 'Bulk action: ' . $bulkAction]);
                        
                        // Add update to application_updates table
                        $updateText = "Status changed from pending to " . $bulkAction . " (bulk action)";
                        $stmt = $pdo->prepare("INSERT INTO application_updates (application_id, user_id, update_text, update_type, created_at) VALUES (?, ?, ?, 'admin', NOW())");
                        $stmt->execute([$appId, $currentUser['id'], $updateText]);
                        
                        $updatedCount++;
                    }
                }
            }
            
            if ($updatedCount > 0) {
                $success = true;
                $successMessage = "Successfully updated $updatedCount application(s)!";
            } else {
                $errors[] = "No applications were updated";
            }
        }
    } catch (Exception $e) {
        error_log("Bulk action error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Get pending applications with user details
try {
    $pdo = getDatabaseConnection();
    
    // Build query with optional filters
    $whereConditions = ["pa.status = 'pending'"];
    $params = [];
    
    // Search filter
    if (!empty($_GET['search'])) {
        $search = '%' . $_GET['search'] . '%';
        $whereConditions[] = "(u.firstname LIKE ? OR u.lastname LIKE ? OR u.email LIKE ? OR pa.university LIKE ? OR pa.program LIKE ?)";
        $params = array_merge($params, [$search, $search, $search, $search, $search]);
    }
    
    // Country filter
    if (!empty($_GET['country'])) {
        $whereConditions[] = "pa.country = ?";
        $params[] = $_GET['country'];
    }
    
    // Degree filter
    if (!empty($_GET['degree'])) {
        $whereConditions[] = "pa.degree = ?";
        $params[] = $_GET['degree'];
    }
    
    $whereClause = implode(' AND ', $whereConditions);
    
    $sql = "SELECT pa.*, u.firstname, u.lastname, u.email, u.phone,
                   (SELECT COUNT(*) FROM application_documents WHERE application_id = pa.id) as document_count,
                   (SELECT COUNT(*) FROM application_updates WHERE application_id = pa.id) as update_count
            FROM portal_applications pa
            JOIN users u ON pa.user_id = u.id
            WHERE $whereClause
            ORDER BY pa.created_at DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $pendingApplications = $stmt->fetchAll();
    
    // Get unique countries and degrees for filters
    $stmt = $pdo->query("SELECT DISTINCT country FROM portal_applications WHERE status = 'pending' ORDER BY country");
    $countries = $stmt->fetchAll();
    
    $stmt = $pdo->query("SELECT DISTINCT degree FROM portal_applications WHERE status = 'pending' ORDER BY degree");
    $degrees = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Pending applications error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pending Applications - Admin Panel</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="admin.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-4 text-lg font-semibold text-gray-900">Admin Panel</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm text-gray-600">
                        Welcome, <?php echo htmlspecialchars($currentUser['firstname'] . ' ' . $currentUser['lastname']); ?>
                        <span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded ml-2">
                            <?php echo UserAuth::getRoleDisplayName($currentUser['role']); ?>
                        </span>
                    </span>
                    <a href="index.php" class="text-blue-600 hover:text-blue-500 text-sm">Portal</a>
                    <a href="logout.php" class="text-red-600 hover:text-red-500 text-sm">Logout</a>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <!-- Page Header -->
        <div class="px-4 py-6 sm:px-0">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Pending Applications</h1>
                    <p class="mt-2 text-gray-600">Review and manage applications awaiting approval</p>
                </div>
                <div class="flex space-x-3">
                    <a href="admin.php" class="text-gray-600 hover:text-gray-800">
                        <i class="fas fa-arrow-left mr-1"></i>
                        Back to Dashboard
                    </a>
                </div>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if ($success): ?>
            <?php ErrorHandler::displaySuccess($successMessage); ?>
        <?php endif; ?>

        <!-- Filters -->
        <div class="bg-white shadow rounded-lg mb-6">
            <div class="px-6 py-4 border-b border-gray-200">
                <h3 class="text-lg font-medium text-gray-900">Filters</h3>
            </div>
            <div class="p-6">
                <form method="GET" action="" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div>
                        <label for="search" class="block text-sm font-medium text-gray-700 mb-1">Search</label>
                        <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>" 
                               class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Name, email, university...">
                    </div>
                    <div>
                        <label for="country" class="block text-sm font-medium text-gray-700 mb-1">Country</label>
                        <select name="country" id="country" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">All Countries</option>
                            <?php foreach ($countries as $country): ?>
                                <option value="<?php echo htmlspecialchars($country['country']); ?>" <?php echo ($_GET['country'] ?? '') === $country['country'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($country['country']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label for="degree" class="block text-sm font-medium text-gray-700 mb-1">Degree</label>
                        <select name="degree" id="degree" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">All Degrees</option>
                            <?php foreach ($degrees as $degree): ?>
                                <option value="<?php echo htmlspecialchars($degree['degree']); ?>" <?php echo ($_GET['degree'] ?? '') === $degree['degree'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars(html_entity_decode($degree['degree'])); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="flex items-end">
                        <button type="submit" class="w-full px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            <i class="fas fa-search mr-1"></i>
                            Filter
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Applications Table -->
        <div class="bg-white shadow rounded-lg">
            <div class="px-6 py-4 border-b border-gray-200">
                <div class="flex items-center justify-between">
                    <h3 class="text-lg font-medium text-gray-900">
                        Pending Applications (<?php echo count($pendingApplications); ?>)
                    </h3>
                    <?php if (!empty($pendingApplications)): ?>
                        <div class="flex space-x-2">
                            <button onclick="selectAll()" class="text-sm text-blue-600 hover:text-blue-800">Select All</button>
                            <button onclick="deselectAll()" class="text-sm text-gray-600 hover:text-gray-800">Deselect All</button>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <?php if (empty($pendingApplications)): ?>
                <div class="p-6 text-center">
                    <i class="fas fa-inbox text-gray-400 text-4xl mb-4"></i>
                    <p class="text-gray-500">No pending applications found</p>
                    <p class="text-sm text-gray-400 mt-1">All applications have been processed or no applications match your filters</p>
                </div>
            <?php else: ?>
                <form method="POST" action="" id="applicationsForm">
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                    
                    <!-- Bulk Actions -->
                    <div class="px-6 py-3 bg-gray-50 border-b border-gray-200">
                        <div class="flex items-center space-x-4">
                            <select name="bulk_action_type" id="bulk_action_type" class="px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                                <option value="">Bulk Actions</option>
                                <option value="in_progress">Mark as In Progress</option>
                                <option value="approved">Approve</option>
                                <option value="rejected">Reject</option>
                            </select>
                            <button type="submit" name="bulk_action" onclick="return confirmBulkAction()" class="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                Apply to Selected
                            </button>
                        </div>
                    </div>
                    
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        <input type="checkbox" id="selectAll" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                    </th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Applicant</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Application</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Documents</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($pendingApplications as $app): ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <input type="checkbox" name="selected_applications[]" value="<?php echo $app['id']; ?>" class="application-checkbox rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div>
                                                <div class="text-sm font-medium text-gray-900">
                                                    <?php echo htmlspecialchars($app['firstname'] . ' ' . $app['lastname']); ?>
                                                </div>
                                                <div class="text-sm text-gray-500">
                                                    <?php echo htmlspecialchars($app['email']); ?>
                                                </div>
                                                <?php if ($app['phone']): ?>
                                                    <div class="text-sm text-gray-500">
                                                        <?php echo htmlspecialchars($app['phone']); ?>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4">
                                            <div>
                                                <div class="text-sm font-medium text-gray-900">
                                                    <?php echo htmlspecialchars($app['university']); ?>
                                                </div>
                                                <div class="text-sm text-gray-500">
                                                    <?php echo htmlspecialchars(html_entity_decode($app['degree'])); ?> in <?php echo htmlspecialchars($app['program']); ?>
                                                </div>
                                                <div class="text-sm text-gray-500">
                                                    <?php echo htmlspecialchars($app['country']); ?> • <?php echo htmlspecialchars($app['intake']); ?>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="text-sm text-gray-900">
                                                <?php echo $app['document_count']; ?> document(s)
                                            </div>
                                            <div class="text-sm text-gray-500">
                                                <?php echo $app['update_count']; ?> update(s)
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                            <?php echo date('M j, Y', strtotime($app['created_at'])); ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                            <div class="flex space-x-2">
                                                <a href="application-details.php?id=<?php echo $app['id']; ?>" class="text-blue-600 hover:text-blue-900" title="View Details">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                <button onclick="openStatusModal(<?php echo $app['id']; ?>)" class="text-green-600 hover:text-green-900" title="Update Status">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <a href="upload-documents.php?application_id=<?php echo $app['id']; ?>" class="text-purple-600 hover:text-purple-900" title="View Documents">
                                                    <i class="fas fa-file-alt"></i>
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>

    <!-- Status Update Modal -->
    <div id="statusModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Update Application Status</h3>
                <form method="POST" action="" id="statusForm">
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                    <input type="hidden" name="application_id" id="modalApplicationId">
                    <input type="hidden" name="update_status" value="1">
                    
                    <div class="mb-4">
                        <label for="new_status" class="block text-sm font-medium text-gray-700 mb-2">New Status</label>
                        <select name="new_status" id="new_status" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">Select status</option>
                            <option value="in_progress">In Progress</option>
                            <option value="approved">Approved</option>
                            <option value="rejected">Rejected</option>
                        </select>
                    </div>
                    
                    <div class="mb-4">
                        <label for="status_reason" class="block text-sm font-medium text-gray-700 mb-2">Reason (Optional)</label>
                        <textarea name="status_reason" id="status_reason" rows="3" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" placeholder="Add a reason for the status change..."></textarea>
                    </div>
                    
                    <div class="flex justify-end space-x-3">
                        <button type="button" onclick="closeStatusModal()" class="px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            Cancel
                        </button>
                        <button type="submit" class="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            Update Status
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        function selectAll() {
            document.querySelectorAll('.application-checkbox').forEach(checkbox => {
                checkbox.checked = true;
            });
            document.getElementById('selectAll').checked = true;
        }
        
        function deselectAll() {
            document.querySelectorAll('.application-checkbox').forEach(checkbox => {
                checkbox.checked = false;
            });
            document.getElementById('selectAll').checked = false;
        }
        
        function confirmBulkAction() {
            const selectedCount = document.querySelectorAll('.application-checkbox:checked').length;
            const action = document.getElementById('bulk_action_type').value;
            
            if (selectedCount === 0) {
                alert('Please select at least one application');
                return false;
            }
            
            if (!action) {
                alert('Please select a bulk action');
                return false;
            }
            
            return confirm(`Are you sure you want to ${action} ${selectedCount} application(s)?`);
        }
        
        function openStatusModal(applicationId) {
            document.getElementById('modalApplicationId').value = applicationId;
            document.getElementById('statusModal').classList.remove('hidden');
        }
        
        function closeStatusModal() {
            document.getElementById('statusModal').classList.add('hidden');
            document.getElementById('statusForm').reset();
        }
        
        // Handle select all checkbox
        document.getElementById('selectAll').addEventListener('change', function() {
            document.querySelectorAll('.application-checkbox').forEach(checkbox => {
                checkbox.checked = this.checked;
            });
        });
        
        // Handle individual checkboxes
        document.querySelectorAll('.application-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const allChecked = document.querySelectorAll('.application-checkbox:checked').length === document.querySelectorAll('.application-checkbox').length;
                document.getElementById('selectAll').checked = allChecked;
            });
        });
        
        // Close modal when clicking outside
        document.getElementById('statusModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeStatusModal();
            }
        });
    </script>
</body>
</html> 