<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get application ID
$applicationId = $_GET['id'] ?? 0;

if (!$applicationId) {
    header('Location: applications.php');
    exit();
}

// Get user data and application
try {
    $pdo = getDatabaseConnection();
    
    // Get user data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    // Get application details from portal_applications table
    $stmt = $pdo->prepare("SELECT * FROM portal_applications WHERE id = ? AND user_id = ?");
    $stmt->execute([$applicationId, $_SESSION['user_id']]);
    $application = $stmt->fetch();
    
    if (!$application) {
        header('Location: applications.php');
        exit();
    }
    
    // Get application updates/timeline
    $stmt = $pdo->prepare("SELECT * FROM application_updates WHERE application_id = ? ORDER BY created_at DESC");
    $stmt->execute([$applicationId]);
    $updates = $stmt->fetchAll();
    
    // Get related documents
    $stmt = $pdo->prepare("SELECT * FROM application_documents WHERE application_id = ? ORDER BY uploaded_at DESC");
    $stmt->execute([$applicationId]);
    $documents = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Application details error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Handle status update submission
if (isset($_POST['add_update'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        $updateText = Validator::validateText($_POST['update_text'] ?? '', 1000);
        
        if (!$updateText) {
            $errors[] = "Please enter an update message";
        } else {
            $stmt = $pdo->prepare("INSERT INTO application_updates (application_id, user_id, update_text, created_at) VALUES (?, ?, ?, NOW())");
            $stmt->execute([$applicationId, $user['id'], $updateText]);
            
            $success = true;
            $successMessage = "Update added successfully!";
            
            // Refresh updates
            $stmt = $pdo->prepare("SELECT * FROM application_updates WHERE application_id = ? ORDER BY created_at DESC");
            $stmt->execute([$applicationId]);
            $updates = $stmt->fetchAll();
        }
    } catch (Exception $e) {
        error_log("Update submission error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();

// Status timeline data
$statusTimeline = [
    'pending' => [
        'icon' => 'fas fa-clock',
        'color' => 'text-yellow-600',
        'bg' => 'bg-yellow-100',
        'title' => 'Application Submitted',
        'description' => 'Your application has been received and is under review'
    ],
    'in_progress' => [
        'icon' => 'fas fa-spinner',
        'color' => 'text-blue-600',
        'bg' => 'bg-blue-100',
        'title' => 'In Progress',
        'description' => 'We are actively working on your application'
    ],
    'completed' => [
        'icon' => 'fas fa-check-circle',
        'color' => 'text-green-600',
        'bg' => 'bg-green-100',
        'title' => 'Completed',
        'description' => 'Your application has been successfully processed'
    ],
    'rejected' => [
        'icon' => 'fas fa-times-circle',
        'color' => 'text-red-600',
        'bg' => 'bg-red-100',
        'title' => 'Rejected',
        'description' => 'Your application was not successful'
    ]
];

$currentStatus = $application['status'] ?? 'pending';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Application Details - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="applications.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-arrow-left"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($user['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Application #<?php echo $application['id']; ?></h1>
                    <p class="text-gray-600 mt-2"><?php echo htmlspecialchars($application['university']); ?> - <?php echo htmlspecialchars(html_entity_decode($application['degree'])); ?></p>
                </div>
                <div class="flex space-x-3">
                    <a href="edit-application.php?id=<?php echo $application['id']; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">
                        <i class="fas fa-edit mr-2"></i>
                        Edit
                    </a>
                    <a href="applications.php" class="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Back to Applications
                    </a>
                </div>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if ($success): ?>
            <?php ErrorHandler::displaySuccess($successMessage); ?>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <!-- Main Content -->
            <div class="lg:col-span-2 space-y-6">
                <!-- Application Details -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Application Details</h2>
                    </div>
                    <div class="p-6">
                        <dl class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <dt class="text-sm font-medium text-gray-500">University</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($application['university']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Country</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($application['country']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Degree Type</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars(html_entity_decode($application['degree'])); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Program</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($application['program'] ?? 'Not specified'); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Intake Period</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($application['intake'] ?? 'Not specified'); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Application Date</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo date('F j, Y', strtotime($application['created_at'])); ?></dd>
                            </div>
                            <?php if (!empty($application['notes'])): ?>
                                <div class="md:col-span-2">
                                    <dt class="text-sm font-medium text-gray-500">Additional Notes</dt>
                                    <dd class="mt-1 text-sm text-gray-900"><?php echo nl2br(htmlspecialchars($application['notes'])); ?></dd>
                                </div>
                            <?php endif; ?>
                        </dl>
                    </div>
                </div>

                <!-- Status Timeline -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Application Status</h2>
                    </div>
                    <div class="p-6">
                        <div class="flow-root">
                            <ul class="-mb-8">
                                <?php foreach ($statusTimeline as $status => $data): ?>
                                    <li>
                                        <div class="relative pb-8">
                                            <?php if ($status !== 'rejected'): ?>
                                                <span class="absolute top-4 left-4 -ml-px h-full w-0.5 bg-gray-200" aria-hidden="true"></span>
                                            <?php endif; ?>
                                            <div class="relative flex space-x-3">
                                                <div>
                                                    <span class="h-8 w-8 rounded-full flex items-center justify-center ring-8 ring-white <?php echo $data['bg']; ?>">
                                                        <i class="<?php echo $data['icon'] . ' ' . $data['color']; ?> text-sm"></i>
                                                    </span>
                                                </div>
                                                <div class="min-w-0 flex-1 pt-1.5 flex justify-between space-x-4">
                                                    <div>
                                                        <p class="text-sm text-gray-500">
                                                            <?php echo $data['title']; ?>
                                                            <?php if ($status === $currentStatus): ?>
                                                                <span class="font-medium text-gray-900"> - Current Status</span>
                                                            <?php endif; ?>
                                                        </p>
                                                        <p class="text-sm text-gray-500"><?php echo $data['description']; ?></p>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    </div>
                </div>

                <!-- Updates and Communication -->
                <div class="bg-white rounded-lg shadow" id="updates">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Updates & Communication</h2>
                    </div>
                    <div class="p-6">
                        <!-- Add Update Form -->
                        <form method="POST" action="" class="mb-6">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                            <div class="mb-4">
                                <label for="update_text" class="block text-sm font-medium text-gray-700 mb-2">Add an Update</label>
                                <textarea
                                    name="update_text"
                                    id="update_text"
                                    rows="3"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                                    placeholder="Add any updates, questions, or additional information..."
                                ></textarea>
                            </div>
                            <button
                                type="submit"
                                name="add_update"
                                class="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700"
                            >
                                <i class="fas fa-plus mr-2"></i>
                                Add Update
                            </button>
                        </form>

                        <!-- Updates List -->
                        <div class="space-y-4">
                            <?php if (empty($updates)): ?>
                                <p class="text-gray-500 text-center py-4">No updates yet. Add the first update above.</p>
                            <?php else: ?>
                                <?php foreach ($updates as $update): ?>
                                    <div class="bg-gray-50 rounded-lg p-4">
                                        <div class="flex items-start space-x-3">
                                            <div class="flex-shrink-0">
                                                <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                                                    <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($user['firstname'], 0, 1)); ?></span>
                                                </div>
                                            </div>
                                            <div class="flex-1 min-w-0">
                                                <p class="text-sm text-gray-900"><?php echo nl2br(htmlspecialchars($update['update_text'])); ?></p>
                                                <p class="text-xs text-gray-500 mt-1">
                                                    <?php echo date('M j, Y g:i A', strtotime($update['created_at'])); ?>
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="space-y-6">
                <!-- Status Card -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Current Status</h3>
                    </div>
                    <div class="p-6">
                        <div class="flex items-center">
                            <div class="flex-shrink-0">
                                <span class="h-10 w-10 rounded-full flex items-center justify-center <?php echo $statusTimeline[$currentStatus]['bg']; ?>">
                                    <i class="<?php echo $statusTimeline[$currentStatus]['icon'] . ' ' . $statusTimeline[$currentStatus]['color']; ?>"></i>
                                </span>
                            </div>
                            <div class="ml-4">
                                <p class="text-sm font-medium text-gray-900"><?php echo $statusTimeline[$currentStatus]['title']; ?></p>
                                <p class="text-sm text-gray-500"><?php echo $statusTimeline[$currentStatus]['description']; ?></p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Documents -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Documents</h3>
                    </div>
                    <div class="p-6">
                        <?php if (empty($documents)): ?>
                            <p class="text-gray-500 text-sm">No documents uploaded yet.</p>
                        <?php else: ?>
                            <div class="space-y-3">
                                <?php foreach ($documents as $doc): ?>
                                    <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                        <div class="flex items-center">
                                            <i class="fas fa-file-alt text-gray-400 mr-3"></i>
                                            <div>
                                                <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($doc['filename']); ?></p>
                                                <p class="text-xs text-gray-500"><?php echo date('M j, Y', strtotime($doc['uploaded_at'])); ?></p>
                                            </div>
                                        </div>
                                        <a href="download-document.php?id=<?php echo $doc['id']; ?>" class="text-blue-600 hover:text-blue-800">
                                            <i class="fas fa-download"></i>
                                        </a>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                        <div class="mt-4">
                            <a href="upload-documents.php?application_id=<?php echo $application['id']; ?>" class="inline-flex items-center px-3 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">
                                <i class="fas fa-upload mr-2"></i>
                                Upload Documents
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Quick Actions</h3>
                    </div>
                    <div class="p-6 space-y-3">
                        <a href="mailto:services@migration.ng?subject=Application #<?php echo $application['id']; ?> - <?php echo htmlspecialchars($application['university']); ?>" class="w-full inline-flex items-center justify-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">
                            <i class="fas fa-envelope mr-2"></i>
                            Contact Support
                        </a>
                        <button onclick="printApplication()" class="w-full inline-flex items-center justify-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">
                            <i class="fas fa-print mr-2"></i>
                            Print Details
                        </button>
                        <?php if ($currentStatus === 'pending'): ?>
                            <button onclick="cancelApplication(<?php echo $application['id']; ?>)" class="w-full inline-flex items-center justify-center px-4 py-2 border border-red-300 rounded-md shadow-sm text-sm font-medium text-red-700 bg-white hover:bg-red-50">
                                <i class="fas fa-times mr-2"></i>
                                Cancel Application
                            </button>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        function printApplication() {
            window.print();
        }
        
        function cancelApplication(applicationId) {
            if (confirm('Are you sure you want to cancel this application? This action cannot be undone.')) {
                $.post('cancel-application.php', {
                    application_id: applicationId,
                    csrf_token: '<?php echo $csrfToken; ?>'
                }).done(function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Failed to cancel application. Please try again.');
                    }
                }).fail(function() {
                    alert('An error occurred. Please try again.');
                });
            }
        }
        
        // Auto-scroll to updates section if URL contains #updates
        if (window.location.hash === '#updates') {
            setTimeout(function() {
                const updatesSection = document.getElementById('updates');
                if (updatesSection) {
                    updatesSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
                }
            }, 100);
        }
    </script>
</body>
</html> 