<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get request ID
$requestId = $_GET['id'] ?? 0;

if (!$requestId) {
    header('Location: document-services.php');
    exit();
}

// Get user data and request details
try {
    $pdo = getDatabaseConnection();
    
    // Get user data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    // Get document request details
    if (UserAuth::isAdmin()) {
        // Admins can view any document request
        $stmt = $pdo->prepare("
            SELECT dr.*, dst.name as service_name, oo.name as office_name, oo.type as office_type
            FROM document_requests dr
            LEFT JOIN document_service_types dst ON dr.service_type_id = dst.id
            LEFT JOIN official_offices oo ON dr.office_id = oo.id
            WHERE dr.id = ?
        ");
        $stmt->execute([$requestId]);
    } else {
        // Regular users can only view their own document requests
        $stmt = $pdo->prepare("
            SELECT dr.*, dst.name as service_name, oo.name as office_name, oo.type as office_type
            FROM document_requests dr
            LEFT JOIN document_service_types dst ON dr.service_type_id = dst.id
            LEFT JOIN official_offices oo ON dr.office_id = oo.id
            WHERE dr.id = ? AND dr.user_id = ?
        ");
        $stmt->execute([$requestId, $_SESSION['user_id']]);
    }
    $request = $stmt->fetch();
    
    if (!$request) {
        header('Location: document-services.php');
        exit();
    }
    
    // If admin is viewing, get the document request owner's information
    if (UserAuth::isAdmin()) {
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$request['user_id']]);
        $requestOwner = $stmt->fetch();
    } else {
        $requestOwner = $user; // For regular users, the owner is themselves
    }
    
    // Get request updates/timeline
    $stmt = $pdo->prepare("
        SELECT dru.*, u.firstname, u.lastname 
        FROM document_request_updates dru
        LEFT JOIN users u ON dru.user_id = u.id
        WHERE dru.request_id = ? 
        ORDER BY dru.created_at DESC
    ");
    $stmt->execute([$requestId]);
    $updates = $stmt->fetchAll();
    
    // Get uploaded documents
    $stmt = $pdo->prepare("
        SELECT * FROM officialisation_documents 
        WHERE request_id = ? 
        ORDER BY uploaded_at DESC
    ");
    $stmt->execute([$requestId]);
    $documents = $stmt->fetchAll();
    
    // Get payments
    $stmt = $pdo->prepare("
        SELECT * FROM document_request_payments 
        WHERE request_id = ? 
        ORDER BY payment_date DESC
    ");
    $stmt->execute([$requestId]);
    $payments = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Document request details error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Handle update submission
if (isset($_POST['add_update'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        $updateText = Validator::validateText($_POST['update_text'] ?? '', 1000);
        
        if (!$updateText) {
            $errors[] = "Please enter an update message";
        } else {
            $updateType = UserAuth::isAdmin() ? 'admin' : 'user';
            $stmt = $pdo->prepare("INSERT INTO document_request_updates (request_id, user_id, update_text, update_type, created_at) VALUES (?, ?, ?, ?, NOW())");
            $stmt->execute([$requestId, $user['id'], $updateText, $updateType]);
            
            // Send email notification
            try {
                if (UserAuth::isAdmin()) {
                    // Admin update - notify the document request owner
                    $subject = "New Update on Document Request #{$request['request_number']} - Migration.ng";
                    $message = "<p>Dear {$requestOwner['firstname']},</p>";
                    $message .= "<p>A new update has been added to your document request #{$request['request_number']} by our admin team.</p>";
                    $message .= "<p><strong>Update:</strong> " . makeLinksClickable(htmlspecialchars($updateText)) . "</p>";
                    $message .= "<p>Please log in to your portal to view the full details.</p>";
                    $message .= "<p>Best regards,<br>Migration.ng Team</p>";
                    
                    EmailService::sendGeneralEmail($requestOwner['email'], $subject, $message);
                } else {
                    // User update - notify admins
                    $stmt = $pdo->prepare("SELECT email, firstname, lastname FROM users WHERE role = 'admin'");
                    $stmt->execute();
                    $admins = $stmt->fetchAll();
                    
                    $subject = "New Update on Document Request #{$request['request_number']} - Migration.ng";
                    $message = "<p>Dear Admin,</p>";
                    $message .= "<p>A new update has been added to document request #{$request['request_number']} by {$user['firstname']} {$user['lastname']}.</p>";
                    $message .= "<p><strong>Update:</strong> " . makeLinksClickable(htmlspecialchars($updateText)) . "</p>";
                    $message .= "<p>You can view the full details in the admin panel.</p>";
                    $message .= "<p>Best regards,<br>Migration.ng System</p>";
                    
                    foreach ($admins as $admin) {
                        EmailService::sendGeneralEmail($admin['email'], $subject, $message);
                    }
                }
            } catch (Exception $e) {
                error_log("Failed to send update notification email: " . $e->getMessage());
            }
            
            $success = true;
            $successMessage = "Update added successfully!";
            
            // Refresh updates
            $stmt = $pdo->prepare("
                SELECT dru.*, u.firstname, u.lastname 
                FROM document_request_updates dru
                LEFT JOIN users u ON dru.user_id = u.id
                WHERE dru.request_id = ? 
                ORDER BY dru.created_at DESC
            ");
            $stmt->execute([$requestId]);
            $updates = $stmt->fetchAll();
        }
    } catch (Exception $e) {
        error_log("Update submission error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();

// Get status color
function getStatusColor($status) {
    switch ($status) {
        case 'pending': return 'bg-yellow-100 text-yellow-800';
        case 'submitted': return 'bg-blue-100 text-blue-800';
        case 'in_progress': return 'bg-orange-100 text-orange-800';
        case 'completed': return 'bg-green-100 text-green-800';
        case 'rejected': return 'bg-red-100 text-red-800';
        case 'cancelled': return 'bg-gray-100 text-gray-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}

// Get urgency color
function getUrgencyColor($urgency) {
    switch ($urgency) {
        case 'normal': return 'bg-green-100 text-green-800';
        case 'urgent': return 'bg-orange-100 text-orange-800';
        case 'express': return 'bg-red-100 text-red-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}

// Convert URLs to clickable links
function makeLinksClickable($text) {
    // Pattern to match URLs
    $urlPattern = '/(https?:\/\/[^\s]+)/i';
    
    // Replace URLs with clickable links
    $text = preg_replace($urlPattern, '<a href="$1" target="_blank" class="text-blue-600 hover:text-blue-800 underline">$1</a>', $text);
    
    // Also handle URLs without http/https
    $urlPattern2 = '/(www\.[^\s]+)/i';
    $text = preg_replace($urlPattern2, '<a href="https://$1" target="_blank" class="text-blue-600 hover:text-blue-800 underline">$1</a>', $text);
    
    return $text;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Document Request Details - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($user['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Document Request Details</h1>
                    <p class="text-gray-600 mt-2">Request #<?php echo htmlspecialchars($request['request_number']); ?></p>
                </div>
                <a href="document-services.php" class="text-gray-600 hover:text-gray-800">
                    <i class="fas fa-arrow-left mr-1"></i>
                    Back to Services
                </a>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if (isset($success) && $success): ?>
            <?php ErrorHandler::displaySuccess($successMessage); ?>
        <?php endif; ?>

        <!-- Success message from URL parameter -->
        <?php if (isset($_GET['success']) && $_GET['success'] == '1'): ?>
            <div class="mb-6 bg-green-50 border border-green-200 rounded-md p-4">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-check-circle text-green-400"></i>
                    </div>
                    <div class="ml-3">
                        <h3 class="text-sm font-medium text-green-800">Document request submitted successfully!</h3>
                        <p class="mt-1 text-sm text-green-700">Your request has been received and is being processed.</p>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Success message from session -->
        <?php if (isset($_SESSION['success'])): ?>
            <div class="mb-6 bg-green-50 border border-green-200 rounded-md p-4">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-check-circle text-green-400"></i>
                    </div>
                    <div class="ml-3">
                        <h3 class="text-sm font-medium text-green-800">Success!</h3>
                        <p class="mt-1 text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['success']); ?></p>
                    </div>
                </div>
            </div>
            <?php unset($_SESSION['success']); ?>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <!-- Main Content -->
            <div class="lg:col-span-2 space-y-6">
                <!-- Request Details -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Request Information</h2>
                    </div>
                    <div class="px-6 py-4">
                        <dl class="grid grid-cols-1 gap-x-4 gap-y-6 sm:grid-cols-2">
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Request Number</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['request_number']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Status</dt>
                                <dd class="mt-1">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo getStatusColor($request['status']); ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $request['status'])); ?>
                                    </span>
                                </dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Service Type</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['service_name']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Urgency Level</dt>
                                <dd class="mt-1">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo getUrgencyColor($request['urgency_level']); ?>">
                                        <?php echo ucfirst($request['urgency_level']); ?>
                                    </span>
                                </dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Document Type</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['document_type']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Destination Country</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['destination_country']); ?></dd>
                            </div>
                            <?php if ($request['office_name']): ?>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Processing Office</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['office_name']); ?></dd>
                            </div>
                            <?php endif; ?>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Total Amount</dt>
                                <dd class="mt-1 text-sm text-gray-900">₦<?php echo number_format($request['total_amount'], 2); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Payment Status</dt>
                                <dd class="mt-1">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $request['payment_status'] === 'paid' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                        <?php echo ucfirst($request['payment_status']); ?>
                                    </span>
                                </dd>
                            </div>
                            <div class="sm:col-span-2">
                                <dt class="text-sm font-medium text-gray-500">Document Description</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo nl2br(htmlspecialchars($request['document_description'])); ?></dd>
                            </div>
                            <?php if ($request['purpose']): ?>
                            <div class="sm:col-span-2">
                                <dt class="text-sm font-medium text-gray-500">Purpose</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['purpose']); ?></dd>
                            </div>
                            <?php endif; ?>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Created</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo date('M j, Y \a\t g:i A', strtotime($request['created_at'])); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Last Updated</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo date('M j, Y \a\t g:i A', strtotime($request['updated_at'])); ?></dd>
                            </div>
                        </dl>
                    </div>
                </div>

                <!-- Applicant Information (Admin View) -->
                <?php if (UserAuth::isAdmin()): ?>
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Applicant Information</h2>
                    </div>
                    <div class="px-6 py-4">
                        <dl class="grid grid-cols-1 gap-x-4 gap-y-6 sm:grid-cols-2">
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Full Name</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($requestOwner['firstname'] . ' ' . $requestOwner['lastname']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Email</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($requestOwner['email']); ?></dd>
                            </div>
                            <?php if ($requestOwner['phone']): ?>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Phone</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($requestOwner['phone']); ?></dd>
                            </div>
                            <?php endif; ?>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">User ID</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($requestOwner['id']); ?></dd>
                            </div>
                        </dl>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Uploaded Documents -->
                <?php if (!empty($documents)): ?>
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Uploaded Documents</h2>
                    </div>
                    <div class="px-6 py-4">
                        <div class="space-y-4">
                            <?php foreach ($documents as $doc): ?>
                            <div class="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                                <div class="flex items-center">
                                    <i class="fas fa-file-alt text-blue-500 mr-3"></i>
                                    <div>
                                        <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($doc['document_name']); ?></p>
                                        <p class="text-sm text-gray-500"><?php echo htmlspecialchars($doc['original_filename']); ?></p>
                                        <p class="text-xs text-gray-400">Uploaded: <?php echo date('M j, Y', strtotime($doc['uploaded_at'])); ?></p>
                                    </div>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <?php if ($doc['is_original']): ?>
                                        <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Original</span>
                                    <?php endif; ?>
                                    <?php if ($doc['is_certified_copy']): ?>
                                        <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">Certified Copy</span>
                                    <?php endif; ?>
                                    <a href="download-document.php?id=<?php echo $doc['id']; ?>" class="text-blue-600 hover:text-blue-900">
                                        <i class="fas fa-download"></i>
                                    </a>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Timeline/Updates -->
                <div class="bg-white rounded-lg shadow" id="updates">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Timeline & Updates</h2>
                    </div>
                    <div class="px-6 py-4">
                        <!-- Add Update Form -->
                        <form method="POST" class="mb-6">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                            <div class="flex space-x-4">
                                <div class="flex-1">
                                    <textarea name="update_text" rows="3" class="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm" placeholder="Add an update or comment..."></textarea>
                                </div>
                                <div class="flex items-end">
                                    <button type="submit" name="add_update" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                        Add Update
                                    </button>
                                </div>
                            </div>
                        </form>

                        <!-- Updates List -->
                        <div class="space-y-4">
                            <?php if (empty($updates)): ?>
                                <p class="text-gray-500 text-center py-4">No updates yet.</p>
                            <?php else: ?>
                                <?php foreach ($updates as $update): ?>
                                <div class="flex space-x-3">
                                    <div class="flex-shrink-0">
                                        <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                            <i class="fas fa-comment text-blue-600 text-sm"></i>
                                        </div>
                                    </div>
                                    <div class="flex-1 min-w-0">
                                        <div class="text-sm">
                                            <span class="font-medium text-gray-900">
                                                <?php echo htmlspecialchars($update['firstname'] . ' ' . $update['lastname']); ?>
                                            </span>
                                            <span class="text-gray-500 ml-2">
                                                <?php echo date('M j, Y \a\t g:i A', strtotime($update['created_at'])); ?>
                                            </span>
                                        </div>
                                        <div class="mt-1 text-sm text-gray-700">
                                            <?php echo nl2br(makeLinksClickable(htmlspecialchars($update['update_text']))); ?>
                                        </div>
                                        <?php if ($update['status_change']): ?>
                                        <div class="mt-1">
                                            <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                                                Status changed to: <?php echo ucfirst(str_replace('_', ' ', $update['status_change'])); ?>
                                            </span>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="space-y-6">
                <!-- Payment Information -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Payment Information</h3>
                    </div>
                    <div class="px-6 py-4">
                        <dl class="space-y-4">
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Total Amount</dt>
                                <dd class="mt-1 text-lg font-semibold text-gray-900">₦<?php echo number_format($request['total_amount'], 2); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Paid Amount</dt>
                                <dd class="mt-1 text-lg font-semibold text-gray-900">₦<?php echo number_format($request['paid_amount'], 2); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Balance</dt>
                                <dd class="mt-1 text-lg font-semibold <?php echo ($request['total_amount'] - $request['paid_amount']) > 0 ? 'text-red-600' : 'text-green-600'; ?>">
                                    ₦<?php echo number_format($request['total_amount'] - $request['paid_amount'], 2); ?>
                                </dd>
                            </div>
                        </dl>
                        
                        <?php if ($request['payment_status'] === 'pending' && $_SESSION['user_id'] == $request['user_id']): ?>
                        <div class="mt-6">
                            <a href="submit-order.php?request_id=<?php echo $request['id']; ?>" class="w-full inline-flex justify-center items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                Submit Order
                            </a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Payment History -->
                <?php if (!empty($payments)): ?>
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Payment History</h3>
                    </div>
                    <div class="px-6 py-4">
                        <div class="space-y-3">
                            <?php foreach ($payments as $payment): ?>
                            <div class="flex justify-between items-center">
                                <div>
                                    <p class="text-sm font-medium text-gray-900">₦<?php echo number_format($payment['amount'], 2); ?></p>
                                    <p class="text-xs text-gray-500"><?php echo date('M j, Y', strtotime($payment['payment_date'])); ?></p>
                                </div>
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                    <?php echo ucfirst($payment['payment_method']); ?>
                                </span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Quick Actions -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Quick Actions</h3>
                    </div>
                    <div class="px-6 py-4 space-y-3">
                        <?php if ($_SESSION['user_id'] == $request['user_id']): ?>
                        <a href="upload-documents.php?request_id=<?php echo $request['id']; ?>" class="w-full inline-flex justify-center items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            <i class="fas fa-upload mr-2"></i>
                            Upload Documents
                        </a>
                        <?php endif; ?>
                        <a href="document-services.php" class="w-full inline-flex justify-center items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            <i class="fas fa-list mr-2"></i>
                            View All Requests
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Auto-hide success messages after 5 seconds
        setTimeout(function() {
            const successMessages = document.querySelectorAll('.bg-green-50');
            successMessages.forEach(function(message) {
                message.style.display = 'none';
            });
        }, 5000);
        
        // Auto-scroll to updates section if URL contains #updates
        if (window.location.hash === '#updates') {
            setTimeout(function() {
                const updatesSection = document.getElementById('updates');
                if (updatesSection) {
                    updatesSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
                }
            }, 100);
        }
    </script>
</body>
</html> 