<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
UserAuth::requireLogin();

// Get user data
$currentUser = UserAuth::getCurrentUser();

// Get service types and pricing
try {
    $pdo = getDatabaseConnection();
    
    // Get active service types
    $stmt = $pdo->query("SELECT * FROM document_service_types WHERE is_active = 1 ORDER BY name");
    $serviceTypes = $stmt->fetchAll();
    
    // Get user's existing document requests
    $stmt = $pdo->prepare("
        SELECT dr.*, dst.name as service_name, oo.name as office_name,
               (SELECT COUNT(*) FROM officialisation_documents WHERE request_id = dr.id) as document_count
        FROM document_requests dr
        JOIN document_service_types dst ON dr.service_type_id = dst.id
        LEFT JOIN official_offices oo ON dr.office_id = oo.id
        WHERE dr.user_id = ?
        ORDER BY dr.created_at DESC
        LIMIT 5
    ");
    $stmt->execute([$currentUser['id']]);
    $recentRequests = $stmt->fetchAll();
    
    // Get user's request statistics
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_requests,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed
        FROM document_requests 
        WHERE user_id = ?
    ");
    $stmt->execute([$currentUser['id']]);
    $requestStats = $stmt->fetch();
    
} catch (Exception $e) {
    error_log("Document services error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Document Services - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($currentUser['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($currentUser['firstname'] . ' ' . $currentUser['lastname']); ?></p>
                            <?php if (UserAuth::isAdmin()): ?>
                                <span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                    Admin
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Document Services</h1>
                    <p class="text-gray-600 mt-2">Legalize and authenticate your documents for international use</p>
                </div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-blue-100 rounded-lg">
                        <i class="fas fa-file-alt text-blue-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Total Requests</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $requestStats['total_requests'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-yellow-100 rounded-lg">
                        <i class="fas fa-clock text-yellow-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Pending</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $requestStats['pending'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-blue-100 rounded-lg">
                        <i class="fas fa-spinner text-blue-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">In Progress</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $requestStats['in_progress'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-green-100 rounded-lg">
                        <i class="fas fa-check-circle text-green-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Completed</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $requestStats['completed'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Service Selection -->
        <div class="bg-white rounded-lg shadow mb-8">
            <div class="px-6 py-4 border-b border-gray-200">
                <h2 class="text-lg font-medium text-gray-900">Select a Service</h2>
                <p class="text-sm text-gray-500 mt-1">Choose from our available document legalization and officialisation services</p>
            </div>
            
            <div class="p-6">
                <!-- Service Dropdown -->
                <div class="mb-6">
                    <label for="service_select" class="block text-sm font-medium text-gray-700 mb-2">Available Services</label>
                    
                    <!-- Custom Dropdown -->
                    <div class="relative">
                        <button type="button" id="dropdown_trigger" class="w-full bg-white border border-gray-300 rounded-md shadow-sm px-4 py-3 text-left focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 hover:border-gray-400 transition-colors">
                            <span id="dropdown_text" class="text-gray-500">Select a service to get started...</span>
                            <span class="absolute inset-y-0 right-0 flex items-center pr-2 pointer-events-none">
                                <i class="fas fa-chevron-down text-gray-400 transition-transform duration-200" id="dropdown_icon"></i>
                            </span>
                        </button>
                        
                        <!-- Dropdown Menu -->
                        <div id="dropdown_menu" class="absolute z-10 w-full mt-1 bg-white border border-gray-200 rounded-md shadow-lg opacity-0 invisible transition-all duration-200 transform scale-95 origin-top">
                            <div class="py-1 max-h-60 overflow-y-auto">
                                <?php foreach ($serviceTypes as $service): ?>
                                    <button type="button" 
                                            class="w-full px-4 py-3 text-left hover:bg-blue-50 focus:bg-blue-50 focus:outline-none transition-colors service-option"
                                            data-value="<?php echo $service['id']; ?>"
                                            data-name="<?php echo htmlspecialchars($service['name']); ?>"
                                            data-description="<?php echo htmlspecialchars($service['description']); ?>"
                                            data-duration="<?php echo htmlspecialchars($service['estimated_duration']); ?>"
                                            data-price="<?php echo $service['base_price']; ?>">
                                        <div class="flex items-center justify-between">
                                            <div class="flex-1">
                                                <div class="font-medium text-gray-900"><?php echo htmlspecialchars($service['name']); ?></div>
                                                <div class="text-sm text-gray-500 mt-1"><?php echo htmlspecialchars($service['description']); ?></div>
                                                <div class="flex items-center text-xs text-gray-400 mt-2">
                                                    <i class="fas fa-clock mr-1"></i>
                                                    <span><?php echo htmlspecialchars($service['estimated_duration']); ?></span>
                                                </div>
                                            </div>
                                            <div class="ml-4 text-right">
                                                <div class="text-lg font-bold text-blue-600">₦<?php echo number_format($service['base_price'], 2); ?></div>
                                                <div class="text-xs text-gray-400">Starting from</div>
                                            </div>
                                        </div>
                                    </button>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Service Summary (hidden by default) -->
                <div id="service_summary" class="hidden bg-gray-50 rounded-lg p-6 border border-gray-200">
                    <div class="flex items-start justify-between">
                        <div class="flex-1">
                            <h3 id="selected_service_name" class="text-lg font-medium text-gray-900 mb-2"></h3>
                            <p id="selected_service_description" class="text-gray-600 mb-4"></p>
                            <div class="flex items-center text-sm text-gray-500">
                                <i class="fas fa-clock mr-2"></i>
                                <span id="selected_service_duration"></span>
                            </div>
                        </div>
                        <div class="text-right ml-6">
                            <div class="text-2xl font-bold text-blue-600" id="selected_service_price"></div>
                            <div class="text-sm text-gray-500">Starting from</div>
                        </div>
                    </div>
                    
                    <div class="mt-6 flex justify-end">
                        <a id="request_service_btn" href="#" class="inline-flex items-center px-6 py-3 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            <i class="fas fa-paper-plane mr-2"></i>
                            Request This Service
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Requests -->
        <div class="bg-white rounded-lg shadow">
            <div class="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
                <h2 class="text-lg font-medium text-gray-900">Recent Requests</h2>
                <a href="document-requests.php" class="text-blue-600 hover:text-blue-800 text-sm font-medium">View All</a>
            </div>
            
            <?php if (empty($recentRequests)): ?>
                <div class="p-8 text-center">
                    <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-gray-100 mb-4">
                        <i class="fas fa-file-alt text-gray-400"></i>
                    </div>
                    <h3 class="text-lg font-medium text-gray-900 mb-2">No document requests yet</h3>
                    <p class="text-gray-500 mb-6">Select a service above to make your first request</p>
                </div>
            <?php else: ?>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Request</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Service</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($recentRequests as $request): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900">
                                            <?php echo htmlspecialchars($request['request_number']); ?>
                                        </div>
                                        <div class="text-sm text-gray-500">
                                            <?php echo htmlspecialchars($request['document_type']); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-900">
                                            <?php echo htmlspecialchars($request['service_name']); ?>
                                        </div>
                                        <?php if ($request['office_name']): ?>
                                            <div class="text-sm text-gray-500">
                                                <?php echo htmlspecialchars($request['office_name']); ?>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php
                                        $status = $request['status'];
                                        $statusClasses = [
                                            'pending' => 'bg-yellow-100 text-yellow-800',
                                            'submitted' => 'bg-blue-100 text-blue-800',
                                            'in_progress' => 'bg-blue-100 text-blue-800',
                                            'completed' => 'bg-green-100 text-green-800',
                                            'rejected' => 'bg-red-100 text-red-800',
                                            'cancelled' => 'bg-gray-100 text-gray-800'
                                        ];
                                        $statusText = [
                                            'pending' => 'Pending',
                                            'submitted' => 'Submitted',
                                            'in_progress' => 'In Progress',
                                            'completed' => 'Completed',
                                            'rejected' => 'Rejected',
                                            'cancelled' => 'Cancelled'
                                        ];
                                        ?>
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?php echo $statusClasses[$status]; ?>">
                                            <?php echo $statusText[$status]; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        ₦<?php echo number_format($request['total_amount'], 2); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo date('M j, Y', strtotime($request['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <a href="document-request-details.php?id=<?php echo $request['id']; ?>" class="text-blue-600 hover:text-blue-900">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>

        <!-- Information Section -->
        <div class="mt-8 grid grid-cols-1 lg:grid-cols-2 gap-8">
            <!-- How It Works -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-medium text-gray-900 mb-4">How It Works</h3>
                <div class="space-y-4">
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <span class="text-blue-600 font-medium text-sm">1</span>
                            </div>
                        </div>
                        <div class="ml-4">
                            <h4 class="text-sm font-medium text-gray-900">Select Service</h4>
                            <p class="text-sm text-gray-500">Choose from our available document legalization services</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <span class="text-blue-600 font-medium text-sm">2</span>
                            </div>
                        </div>
                        <div class="ml-4">
                            <h4 class="text-sm font-medium text-gray-900">Submit Request</h4>
                            <p class="text-sm text-gray-500">Fill out the form and upload your documents</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <span class="text-blue-600 font-medium text-sm">3</span>
                            </div>
                        </div>
                        <div class="ml-4">
                            <h4 class="text-sm font-medium text-gray-900">Payment & Processing</h4>
                            <p class="text-sm text-gray-500">Make payment and we'll process your documents</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <span class="text-blue-600 font-medium text-sm">4</span>
                            </div>
                        </div>
                        <div class="ml-4">
                            <h4 class="text-sm font-medium text-gray-900">Track & Receive</h4>
                            <p class="text-sm text-gray-500">Monitor progress and collect your legalized documents</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Service Benefits -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Why Choose Our Services?</h3>
                <div class="space-y-4">
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <i class="fas fa-check-circle text-green-500 mt-1"></i>
                        </div>
                        <div class="ml-3">
                            <h4 class="text-sm font-medium text-gray-900">Expert Guidance</h4>
                            <p class="text-sm text-gray-500">Professional assistance throughout the entire process</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <i class="fas fa-check-circle text-green-500 mt-1"></i>
                        </div>
                        <div class="ml-3">
                            <h4 class="text-sm font-medium text-gray-900">Fast Processing</h4>
                            <p class="text-sm text-gray-500">Expedited services available for urgent requirements</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <i class="fas fa-check-circle text-green-500 mt-1"></i>
                        </div>
                        <div class="ml-3">
                            <h4 class="text-sm font-medium text-gray-900">Secure Handling</h4>
                            <p class="text-sm text-gray-500">Safe and confidential document processing</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <i class="fas fa-check-circle text-green-500 mt-1"></i>
                        </div>
                        <div class="ml-3">
                            <h4 class="text-sm font-medium text-gray-900">Real-time Updates</h4>
                            <p class="text-sm text-gray-500">Track your request status online anytime</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <i class="fas fa-check-circle text-green-500 mt-1"></i>
                        </div>
                        <div class="ml-3">
                            <h4 class="text-sm font-medium text-gray-900">Nationwide Coverage</h4>
                            <p class="text-sm text-gray-500">Services available across Nigeria and international offices</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Service data from PHP
        const serviceTypes = <?php echo json_encode($serviceTypes); ?>;
        
        // Handle service selection
        document.addEventListener('DOMContentLoaded', function() {
            const dropdownTrigger = document.getElementById('dropdown_trigger');
            const dropdownMenu = document.getElementById('dropdown_menu');
            const dropdownText = document.getElementById('dropdown_text');
            const dropdownIcon = document.getElementById('dropdown_icon');
            const serviceSummary = document.getElementById('service_summary');
            const requestServiceBtn = document.getElementById('request_service_btn');

            // Function to toggle dropdown visibility
            function toggleDropdown() {
                dropdownMenu.classList.toggle('invisible');
                dropdownMenu.classList.toggle('opacity-0');
                dropdownMenu.classList.toggle('scale-95');
                dropdownIcon.classList.toggle('rotate-180');
            }

            // Function to close dropdown when clicking outside
            document.addEventListener('click', function(event) {
                if (!dropdownTrigger.contains(event.target) && !dropdownMenu.contains(event.target)) {
                    dropdownMenu.classList.add('invisible');
                    dropdownMenu.classList.add('opacity-0');
                    dropdownMenu.classList.add('scale-95');
                    dropdownIcon.classList.remove('rotate-180');
                }
            });

            // Handle dropdown trigger click
            dropdownTrigger.addEventListener('click', toggleDropdown);

            // Handle service option clicks
            dropdownMenu.querySelectorAll('.service-option').forEach(option => {
                option.addEventListener('click', function() {
                    const selectedOption = this;
                    dropdownText.textContent = selectedOption.dataset.name;
                    dropdownIcon.classList.remove('rotate-180'); // Close dropdown

                    // Show summary
                    document.getElementById('selected_service_name').textContent = selectedOption.dataset.name;
                    document.getElementById('selected_service_description').textContent = selectedOption.dataset.description;
                    document.getElementById('selected_service_duration').textContent = selectedOption.dataset.duration;
                    document.getElementById('selected_service_price').textContent = '₦' + parseFloat(selectedOption.dataset.price).toLocaleString();
                    
                    // Update request button href
                    requestServiceBtn.href = 'new-document-request.php?service_id=' + selectedOption.dataset.value;
                    
                    // Show summary section
                    serviceSummary.classList.remove('hidden');
                    
                    // Smooth scroll to summary
                    serviceSummary.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                    
                    // Close dropdown
                    dropdownMenu.classList.add('invisible');
                    dropdownMenu.classList.add('opacity-0');
                    dropdownMenu.classList.add('scale-95');
                });
            });

            // Keyboard navigation support
            dropdownTrigger.addEventListener('keydown', function(event) {
                if (event.key === 'Enter' || event.key === ' ') {
                    event.preventDefault();
                    toggleDropdown();
                }
            });

            // Add focus styles
            dropdownTrigger.addEventListener('focus', function() {
                this.classList.add('ring-2', 'ring-blue-500', 'border-blue-500');
            });

            dropdownTrigger.addEventListener('blur', function() {
                this.classList.remove('ring-2', 'ring-blue-500', 'border-blue-500');
            });
        });
    </script>
</body>
</html> 