<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get user data
try {
    $pdo = getDatabaseConnection();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    // Get user's applications for document association
    $stmt = $pdo->prepare("SELECT id, university, degree, program FROM portal_applications WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$_SESSION['user_id']]);
    $applications = $stmt->fetchAll();
    
    // Get user's documents
    $stmt = $pdo->prepare("
        SELECT d.*, a.university, a.degree 
        FROM application_documents d 
        LEFT JOIN portal_applications a ON d.application_id = a.id 
        WHERE d.user_id = ? 
        ORDER BY d.uploaded_at DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $documents = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Documents error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Handle document upload
if (isset($_POST['upload_documents'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        $applicationId = $_POST['application_id'] ?? null;
        $documentType = Validator::validateText($_POST['document_type'] ?? '', 50);
        
        if (!$documentType) {
            $errors[] = "Please select a document type";
        }
        
        // If no validation errors, proceed with file upload
        if (empty($errors)) {
            $uploadDir = '../uploads/' . $user['firstname'] . '_' . $user['lastname'] . '/';
            
            // Create directory if it doesn't exist
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $uploadSuccessCount = 0;
            $uploadErrors = [];
            
            // Process file uploads
            if (!empty(array_filter($_FILES['documents']['name']))) {
                foreach ($_FILES['documents']['tmp_name'] as $key => $value) {
                    $file = [
                        'name' => $_FILES['documents']['name'][$key],
                        'tmp_name' => $_FILES['documents']['tmp_name'][$key],
                        'size' => $_FILES['documents']['size'][$key],
                        'error' => $_FILES['documents']['error'][$key]
                    ];
                    
                    $validatedFile = Validator::validateFile($file);
                    
                    if ($validatedFile) {
                        $safeFilename = Validator::generateSafeFilename($validatedFile['name']);
                        $filepath = $uploadDir . $safeFilename;
                        
                        if (move_uploaded_file($validatedFile['tmp_name'], $filepath)) {
                            // Save document record to database
                            $stmt = $pdo->prepare("INSERT INTO application_documents (user_id, application_id, document_type, filename, original_filename, file_size, file_path, uploaded_at) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())");
                            $stmt->execute([
                                $_SESSION['user_id'],
                                $applicationId,
                                $documentType,
                                $safeFilename,
                                $validatedFile['name'],
                                $validatedFile['size'],
                                $filepath
                            ]);
                            
                            $uploadSuccessCount++;
                        } else {
                            $uploadErrors[] = "Failed to upload " . $validatedFile['name'];
                        }
                    } else {
                        $uploadErrors[] = "Invalid file: " . $file['name'];
                    }
                }
            }
            
            if ($uploadSuccessCount > 0) {
                $success = true;
                $successMessage = "Successfully uploaded $uploadSuccessCount document(s).";
                if (!empty($uploadErrors)) {
                    $successMessage .= " Some files failed to upload: " . implode(', ', $uploadErrors);
                }
                
                // Refresh documents list
                $stmt = $pdo->prepare("
                    SELECT d.*, a.university, a.degree 
                    FROM application_documents d 
                    LEFT JOIN portal_applications a ON d.application_id = a.id 
                    WHERE d.user_id = ? 
                    ORDER BY d.uploaded_at DESC
                ");
                $stmt->execute([$_SESSION['user_id']]);
                $documents = $stmt->fetchAll();
            } else {
                $errors[] = "No documents were uploaded successfully.";
            }
        }
    } catch (Exception $e) {
        error_log("Document upload error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();

// Document types
$documentTypes = [
    'transcript' => 'Academic Transcript',
    'certificate' => 'Certificate/Diploma',
    'passport' => 'Passport',
    'cv' => 'CV/Resume',
    'personal_statement' => 'Personal Statement',
    'reference_letter' => 'Reference Letter',
    'english_test' => 'English Test Result',
    'financial_statement' => 'Financial Statement',
    'other' => 'Other Document'
];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Documents - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($user['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Document Management</h1>
                    <p class="text-gray-600 mt-2">Upload and manage your application documents</p>
                </div>
                <button onclick="openUploadModal()" class="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                    <i class="fas fa-upload mr-2"></i>
                    Upload Documents
                </button>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if ($success): ?>
            <?php ErrorHandler::displaySuccess($successMessage); ?>
        <?php endif; ?>

        <!-- Document Statistics -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-blue-100 rounded-lg">
                        <i class="fas fa-file-alt text-blue-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Total Documents</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo count($documents); ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-green-100 rounded-lg">
                        <i class="fas fa-check-circle text-green-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Uploaded Today</p>
                        <p class="text-2xl font-bold text-gray-900">
                            <?php 
                            $todayCount = 0;
                            foreach ($documents as $doc) {
                                if (date('Y-m-d', strtotime($doc['uploaded_at'])) === date('Y-m-d')) {
                                    $todayCount++;
                                }
                            }
                            echo $todayCount;
                            ?>
                        </p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-purple-100 rounded-lg">
                        <i class="fas fa-folder text-purple-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Active Applications</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo count($applications); ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-yellow-100 rounded-lg">
                        <i class="fas fa-clock text-yellow-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Storage Used</p>
                        <p class="text-2xl font-bold text-gray-900">
                            <?php 
                            $totalSize = 0;
                            foreach ($documents as $doc) {
                                $totalSize += $doc['file_size'];
                            }
                            echo round($totalSize / 1024 / 1024, 1) . ' MB';
                            ?>
                        </p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Documents List -->
        <div class="bg-white rounded-lg shadow">
            <div class="px-6 py-4 border-b border-gray-200">
                <h2 class="text-lg font-medium text-gray-900">Your Documents</h2>
            </div>
            
            <?php if (empty($documents)): ?>
                <div class="p-8 text-center">
                    <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-gray-100 mb-4">
                        <i class="fas fa-file-alt text-gray-400"></i>
                    </div>
                    <h3 class="text-lg font-medium text-gray-900 mb-2">No documents uploaded yet</h3>
                    <p class="text-gray-500 mb-6">Start by uploading your application documents to keep them organized.</p>
                    <button onclick="openUploadModal()" class="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700">
                        <i class="fas fa-upload mr-2"></i>
                        Upload First Document
                    </button>
                </div>
            <?php else: ?>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Document</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Application</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Size</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Uploaded</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($documents as $doc): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="flex-shrink-0 h-10 w-10">
                                                <div class="h-10 w-10 rounded-lg bg-gray-100 flex items-center justify-center">
                                                    <i class="fas fa-file-alt text-gray-400"></i>
                                                </div>
                                            </div>
                                            <div class="ml-4">
                                                <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($doc['original_filename']); ?></div>
                                                <div class="text-sm text-gray-500"><?php echo htmlspecialchars($doc['filename']); ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800">
                                            <?php echo htmlspecialchars($documentTypes[$doc['document_type']] ?? $doc['document_type']); ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php if ($doc['application_id']): ?>
                                            <div class="text-sm text-gray-900"><?php echo htmlspecialchars($doc['university']); ?></div>
                                            <div class="text-sm text-gray-500"><?php echo htmlspecialchars(html_entity_decode($doc['degree'])); ?></div>
                                        <?php else: ?>
                                            <span class="text-sm text-gray-500">Not assigned</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo round($doc['file_size'] / 1024, 1); ?> KB
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo date('M j, Y', strtotime($doc['uploaded_at'])); ?>
                                        <div class="text-xs text-gray-400">
                                            <?php echo date('g:i A', strtotime($doc['uploaded_at'])); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <div class="flex space-x-2">
                                            <a href="download-document.php?id=<?php echo $doc['id']; ?>" class="text-blue-600 hover:text-blue-900" title="Download">
                                                <i class="fas fa-download"></i>
                                            </a>
                                            <a href="preview-document.php?id=<?php echo $doc['id']; ?>" class="text-green-600 hover:text-green-900" title="Preview">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <button onclick="deleteDocument(<?php echo $doc['id']; ?>)" class="text-red-600 hover:text-red-900" title="Delete">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Upload Modal -->
    <div id="uploadModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-medium text-gray-900">Upload Documents</h3>
                    <button onclick="closeUploadModal()" class="text-gray-400 hover:text-gray-600">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <form method="POST" action="" enctype="multipart/form-data">
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                    
                    <div class="mb-4">
                        <label for="document_type" class="block text-sm font-medium text-gray-700 mb-2">Document Type *</label>
                        <select name="document_type" id="document_type" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">Select document type</option>
                            <?php foreach ($documentTypes as $key => $name): ?>
                                <option value="<?php echo $key; ?>"><?php echo htmlspecialchars($name); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-4">
                        <label for="application_id" class="block text-sm font-medium text-gray-700 mb-2">Associate with Application (Optional)</label>
                        <select name="application_id" id="application_id" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">No specific application</option>
                            <?php foreach ($applications as $app): ?>
                                <option value="<?php echo $app['id']; ?>">
                                    <?php echo htmlspecialchars($app['university'] . ' - ' . html_entity_decode($app['degree'])); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-4">
                        <label for="documents" class="block text-sm font-medium text-gray-700 mb-2">Select Files *</label>
                        <input
                            type="file"
                            name="documents[]"
                            id="documents"
                            multiple
                            required
                            accept=".pdf,.doc,.docx,.jpg,.jpeg,.png"
                            class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                        />
                        <p class="text-xs text-gray-500 mt-1">PDF, DOC, DOCX, JPG, PNG (Max 5MB per file)</p>
                    </div>
                    
                    <div class="flex justify-end space-x-3">
                        <button
                            type="button"
                            onclick="closeUploadModal()"
                            class="px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50"
                        >
                            Cancel
                        </button>
                        <button
                            type="submit"
                            name="upload_documents"
                            class="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700"
                        >
                            <i class="fas fa-upload mr-2"></i>
                            Upload
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        function openUploadModal() {
            document.getElementById('uploadModal').classList.remove('hidden');
        }
        
        function closeUploadModal() {
            document.getElementById('uploadModal').classList.add('hidden');
        }
        
        function deleteDocument(documentId) {
            if (confirm('Are you sure you want to delete this document? This action cannot be undone.')) {
                $.post('delete-document.php', {
                    document_id: documentId,
                    csrf_token: '<?php echo $csrfToken; ?>'
                }).done(function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Failed to delete document. Please try again.');
                    }
                }).fail(function() {
                    alert('An error occurred. Please try again.');
                });
            }
        }
        
        // Close modal when clicking outside
        document.getElementById('uploadModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeUploadModal();
            }
        });
    </script>
</body>
</html> 