<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';
require_once '../arrays/countries.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
UserAuth::requireLogin();

// Get user data
$currentUser = UserAuth::getCurrentUser();

// Get application ID from URL
$applicationId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$applicationId) {
    header('Location: applications.php?error=invalid_id');
    exit();
}

// Get application data
try {
    $pdo = getDatabaseConnection();
    $stmt = $pdo->prepare("SELECT * FROM portal_applications WHERE id = ? AND user_id = ?");
    $stmt->execute([$applicationId, $currentUser['id']]);
    $application = $stmt->fetch();
    
    if (!$application) {
        header('Location: applications.php?error=not_found');
        exit();
    }
} catch (Exception $e) {
    error_log("Edit application error: " . $e->getMessage());
    header('Location: applications.php?error=database_error');
    exit();
}

// Initialize variables
$errors = [];
$success = false;

// Handle form submission
if (isset($_POST['update_application'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        // Validate and sanitize input
        $country = Validator::validateCountry($_POST['country'] ?? '');
        $university = Validator::validateText($_POST['university'] ?? '', 100);
        $degree = Validator::validateText($_POST['degree'] ?? '', 50);
        $program = Validator::validateText($_POST['program'] ?? '', 100);
        $intake = Validator::validateText($_POST['intake'] ?? '', 20);
        $notes = Validator::validateText($_POST['notes'] ?? '', 500);
        
        // Check for validation errors
        if (!$country) $errors[] = "Please select a valid country";
        if (!$university) $errors[] = "Please enter the university name";
        if (!$degree) $errors[] = "Please select the degree type";
        if (!$program) $errors[] = "Please enter the program of study";
        if (!$intake) $errors[] = "Please select the intake period";
        
        // If no validation errors, proceed with database update
        if (empty($errors)) {
            $updateSql = "UPDATE portal_applications SET country = ?, university = ?, degree = ?, program = ?, intake = ?, notes = ?, updated_at = NOW() WHERE id = ? AND user_id = ?";
            $stmt = $pdo->prepare($updateSql);
            $stmt->execute([$country, $university, $degree, $program, $intake, $notes, $applicationId, $currentUser['id']]);
            
            // Add update to application_updates table
            $updateText = "Application details updated by user";
            $stmt = $pdo->prepare("INSERT INTO application_updates (application_id, user_id, update_text, update_type, created_at) VALUES (?, ?, ?, 'user', NOW())");
            $stmt->execute([$applicationId, $currentUser['id'], $updateText]);
            
            $success = true;
            $successMessage = "Application updated successfully!";
            
            // Refresh application data
            $stmt = $pdo->prepare("SELECT * FROM portal_applications WHERE id = ? AND user_id = ?");
            $stmt->execute([$applicationId, $currentUser['id']]);
            $application = $stmt->fetch();
        }
    } catch (Exception $e) {
        error_log("Application update error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Application - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        #university-suggestions {
            border: 1px solid #d1d5db;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
        }
        
        #university-suggestions p {
            padding: 12px 16px;
            margin: 0;
            cursor: pointer;
            border-bottom: 1px solid #f3f4f6;
            transition: all 0.2s ease-in-out;
            font-size: 14px;
            line-height: 1.4;
        }
        
        #university-suggestions p:last-child {
            border-bottom: none;
        }
        
        #university-suggestions p:hover {
            background-color: #f8fafc;
            color: #1f2937;
            padding-left: 20px;
        }
        
        #university-suggestions p:active {
            background-color: #e5e7eb;
        }
        
        /* Scrollbar styling for the suggestions dropdown */
        #university-suggestions::-webkit-scrollbar {
            width: 6px;
        }
        
        #university-suggestions::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 3px;
        }
        
        #university-suggestions::-webkit-scrollbar-thumb {
            background: #c1c1c1;
            border-radius: 3px;
        }
        
        #university-suggestions::-webkit-scrollbar-thumb:hover {
            background: #a8a8a8;
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($currentUser['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($currentUser['firstname'] . ' ' . $currentUser['lastname']); ?></p>
                            <?php if (UserAuth::isAdmin()): ?>
                                <span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                    Admin
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Edit Application</h1>
                    <p class="text-gray-600 mt-2">Update your university application details</p>
                    <div class="mt-2 text-sm text-gray-500">
                        Application ID: #<?php echo $application['id']; ?> | 
                        Status: <span class="font-medium text-<?php echo $application['status'] === 'pending' ? 'yellow' : ($application['status'] === 'completed' ? 'green' : 'blue'); ?>-600">
                            <?php echo ucfirst(str_replace('_', ' ', $application['status'])); ?>
                        </span>
                    </div>
                </div>
                <div class="flex space-x-3">
                    <a href="application-details.php?id=<?php echo $application['id']; ?>" class="text-blue-600 hover:text-blue-800">
                        <i class="fas fa-eye mr-1"></i>
                        View Details
                    </a>
                    <a href="applications.php" class="text-gray-600 hover:text-gray-800">
                        <i class="fas fa-arrow-left mr-1"></i>
                        Back to Applications
                    </a>
                </div>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if ($success): ?>
            <?php ErrorHandler::displaySuccess($successMessage); ?>
        <?php endif; ?>

        <!-- Application Form -->
        <div class="bg-white rounded-lg shadow">
            <div class="px-6 py-4 border-b border-gray-200">
                <h2 class="text-lg font-medium text-gray-900">Application Details</h2>
                <p class="text-sm text-gray-500 mt-1">Update the information below and click Save Changes</p>
            </div>
            
            <form method="POST" action="" class="p-6">
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <!-- Country Selection -->
                    <div>
                        <label for="country" class="block text-sm font-medium text-gray-700 mb-2">Country *</label>
                        <select name="country" id="country" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">Select a country</option>
                            <?php foreach ($countries as $code => $name): ?>
                                <option value="<?php echo htmlspecialchars($code); ?>" <?php echo $application['country'] === $code ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($name); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Degree Type -->
                    <div>
                        <label for="degree" class="block text-sm font-medium text-gray-700 mb-2">Degree Type *</label>
                        <select name="degree" id="degree" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">Select degree type</option>
                            <option value="Bachelor's" <?php echo $application['degree'] === "Bachelor's" ? 'selected' : ''; ?>>Bachelor's Degree</option>
                            <option value="Master's" <?php echo $application['degree'] === "Master's" ? 'selected' : ''; ?>>Master's Degree</option>
                            <option value="PhD" <?php echo $application['degree'] === 'PhD' ? 'selected' : ''; ?>>PhD/Doctorate</option>
                            <option value="Diploma" <?php echo $application['degree'] === 'Diploma' ? 'selected' : ''; ?>>Diploma</option>
                            <option value="Certificate" <?php echo $application['degree'] === 'Certificate' ? 'selected' : ''; ?>>Certificate</option>
                            <option value="Foundation" <?php echo $application['degree'] === 'Foundation' ? 'selected' : ''; ?>>Foundation Year</option>
                            <option value="Pre-Master's" <?php echo $application['degree'] === "Pre-Master's" ? 'selected' : ''; ?>>Pre-Master's</option>
                        </select>
                    </div>
                </div>

                <!-- University Selection -->
                <div class="mt-6">
                    <label for="university" class="block text-sm font-medium text-gray-700 mb-2">University *</label>
                    <div class="relative">
                        <input
                            type="text"
                            name="university"
                            id="university"
                            required
                            value="<?php echo htmlspecialchars($application['university']); ?>"
                            class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Start typing university name..."
                            autocomplete="off"
                        />
                        <div id="university-suggestions" class="absolute z-10 w-full mt-1 bg-white border border-gray-300 rounded-md shadow-lg hidden max-h-60 overflow-y-auto">
                            <!-- University suggestions will be populated here -->
                        </div>
                    </div>
                    <p class="text-xs text-gray-500 mt-1">Start typing to search for universities</p>
                </div>

                <!-- Program and Intake -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                    <div>
                        <label for="program" class="block text-sm font-medium text-gray-700 mb-2">Program of Study *</label>
                        <input
                            type="text"
                            name="program"
                            id="program"
                            required
                            value="<?php echo htmlspecialchars($application['program']); ?>"
                            class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                            placeholder="e.g., Computer Science, Business Administration"
                        />
                    </div>

                    <div>
                        <label for="intake" class="block text-sm font-medium text-gray-700 mb-2">Intake Period *</label>
                        <select name="intake" id="intake" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">Select intake</option>
                            <option value="Fall 2024" <?php echo $application['intake'] === 'Fall 2024' ? 'selected' : ''; ?>>Fall 2024</option>
                            <option value="Spring 2025" <?php echo $application['intake'] === 'Spring 2025' ? 'selected' : ''; ?>>Spring 2025</option>
                            <option value="Summer 2025" <?php echo $application['intake'] === 'Summer 2025' ? 'selected' : ''; ?>>Summer 2025</option>
                            <option value="Fall 2025" <?php echo $application['intake'] === 'Fall 2025' ? 'selected' : ''; ?>>Fall 2025</option>
                            <option value="Spring 2026" <?php echo $application['intake'] === 'Spring 2026' ? 'selected' : ''; ?>>Spring 2026</option>
                            <option value="Summer 2026" <?php echo $application['intake'] === 'Summer 2026' ? 'selected' : ''; ?>>Summer 2026</option>
                            <option value="Fall 2026" <?php echo $application['intake'] === 'Fall 2026' ? 'selected' : ''; ?>>Fall 2026</option>
                        </select>
                    </div>
                </div>

                <!-- Additional Notes -->
                <div class="mt-6">
                    <label for="notes" class="block text-sm font-medium text-gray-700 mb-2">Additional Notes</label>
                    <textarea
                        name="notes"
                        id="notes"
                        rows="4"
                        class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Any additional information about your application, specific requirements, or preferences..."
                    ><?php echo htmlspecialchars($application['notes']); ?></textarea>
                    <p class="text-xs text-gray-500 mt-1">Optional: Add any specific requirements or preferences</p>
                </div>

                <!-- Form Actions -->
                <div class="mt-8 flex justify-end space-x-4">
                    <a href="applications.php" class="px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                        Cancel
                    </a>
                    <button
                        type="submit"
                        name="update_application"
                        class="px-6 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                    >
                        <i class="fas fa-save mr-2"></i>
                        Save Changes
                    </button>
                </div>
            </form>
        </div>

        <!-- Information Box -->
        <div class="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-info-circle text-blue-400"></i>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-blue-800">Important Notes</h3>
                    <div class="mt-2 text-sm text-blue-700">
                        <ul class="list-disc list-inside space-y-1">
                            <li>Changes to your application will be logged in the application timeline</li>
                            <li>Our team will be notified of any updates you make</li>
                            <li>You can only edit applications that are still in pending status</li>
                            <li>Some fields may be locked if your application is already being processed</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // University search functionality
        $(document).ready(function() {
            let searchTimeout;
            
            $('#university').on('input', function() {
                const query = $(this).val();
                const country = $('#country').val();
                
                clearTimeout(searchTimeout);
                
                if (query.length < 2) {
                    $('#university-suggestions').addClass('hidden').empty();
                    return;
                }
                
                searchTimeout = setTimeout(function() {
                    $.get('../backend-search.php', {
                        term: query,
                        termCountry: country
                    }).done(function(data) {
                        if (data.trim()) {
                            $('#university-suggestions').html(data).removeClass('hidden');
                        } else {
                            $('#university-suggestions').addClass('hidden').empty();
                        }
                    });
                }, 300);
            });
            
            // Hide suggestions when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('#university, #university-suggestions').length) {
                    $('#university-suggestions').addClass('hidden');
                }
            });
            
            // Select university from suggestions
            $(document).on('click', '#university-suggestions p', function() {
                $('#university').val($(this).text());
                $('#university-suggestions').addClass('hidden');
            });
            
            // Form validation
            $('form').on('submit', function(e) {
                const requiredFields = ['country', 'university', 'degree', 'program', 'intake'];
                let isValid = true;
                
                requiredFields.forEach(function(field) {
                    const value = $('#' + field).val().trim();
                    if (!value) {
                        $('#' + field).addClass('border-red-500');
                        isValid = false;
                    } else {
                        $('#' + field).removeClass('border-red-500');
                    }
                });
                
                if (!isValid) {
                    e.preventDefault();
                    alert('Please fill in all required fields.');
                }
            });
        });
    </script>
</body>
</html> 