<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../../config/database.php';
require_once '../../includes/Validator.php';
require_once '../../includes/ErrorHandler.php';
require_once '../../includes/EmailService.php';
require_once '../../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || !UserAuth::isAdmin()) {
    header('Location: ../login.php');
    exit();
}

// Get admin data
try {
    $pdo = getDatabaseConnection();
    
    // Get admin data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $admin = $stmt->fetch();
    
    if (!$admin) {
        session_destroy();
        header('Location: ../login.php');
        exit();
    }
    
    // Get all users (applicants)
    $stmt = $pdo->prepare("
        SELECT u.*, 
               COUNT(DISTINCT pa.id) as application_count,
               COUNT(DISTINCT dr.id) as document_request_count
        FROM users u
        LEFT JOIN portal_applications pa ON u.id = pa.user_id
        LEFT JOIN document_requests dr ON u.id = dr.user_id
        WHERE u.role != 'admin'
        GROUP BY u.id
        ORDER BY u.created_at DESC
    ");
    $stmt->execute();
    $applicants = $stmt->fetchAll();
    
    // Handle email sending
    $success = false;
    $error = '';
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['send_email'])) {
        try {
            // Validate CSRF token
            if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
                throw new Exception('Invalid request. Please try again.');
            }
            
            $recipientEmail = Validator::validateEmail($_POST['recipient_email'] ?? '');
            $subject = Validator::validateText($_POST['subject'] ?? '', 200);
            $message = Validator::validateText($_POST['message'] ?? '', 5000);
            
            if (!$recipientEmail) {
                throw new Exception('Please enter a valid recipient email address.');
            }
            
            if (!$subject) {
                throw new Exception('Please enter a subject for the email.');
            }
            
            if (!$message) {
                throw new Exception('Please enter a message.');
            }
            
                         // Wrap the message in a proper HTML email template
             $htmlContent = buildStyledEmailTemplate($subject, $message);
             
             // Send the email using EmailService
             $success = EmailService::sendGeneralEmail($recipientEmail, $subject, $htmlContent);
            
            if ($success) {
                $successMessage = "Email sent successfully to $recipientEmail";
            } else {
                throw new Exception('Failed to send email. Please try again.');
            }
            
        } catch (Exception $e) {
            $error = $e->getMessage();
        }
    }
    
} catch (Exception $e) {
    error_log("Email interface error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();

/**
 * Build a styled HTML email template
 * @param string $subject
 * @param string $message
 * @return string
 */
function buildStyledEmailTemplate($subject, $message) {
    $html = "<!DOCTYPE html>
    <html>
    <head>
        <meta charset='UTF-8'>
        <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        <title>" . htmlspecialchars($subject) . "</title>
        <style>
            * { margin: 0; padding: 0; box-sizing: border-box; }
            body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background-color: #f8fafc; }
            .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1); }
                         .header { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); color: white; padding: 40px 30px; text-align: center; }
             .header h1 { font-size: 28px; font-weight: 600; margin-bottom: 10px; }
             .header p { font-size: 16px; opacity: 0.9; }
             .logo { width: 120px; height: auto; margin-bottom: 20px; }
            .content { padding: 40px 30px; background: #ffffff; }
            .message-content { background: #f8fafc; border-radius: 8px; padding: 25px; margin-bottom: 25px; line-height: 1.8; }
            .message-content p { margin-bottom: 15px; }
            .message-content p:last-child { margin-bottom: 0; }
            .message-content ul, .message-content ol { margin: 15px 0; padding-left: 20px; }
            .message-content li { margin-bottom: 8px; }
            .message-content strong { color: #1e293b; font-weight: 600; }
            .message-content em { color: #64748b; }
            .footer { background: #f1f5f9; padding: 30px; text-align: center; color: #64748b; font-size: 14px; }
            .footer a { color: #3b82f6; text-decoration: none; }
            .footer a:hover { text-decoration: underline; }
            @media (max-width: 600px) {
                .container { margin: 10px; border-radius: 8px; }
                .header { padding: 30px 20px; }
                .content { padding: 30px 20px; }
            }
        </style>
    </head>
    <body>
                 <div class='container'>
             <div class='header'>
                 <h1>Migration.ng</h1>
                 <p>Official Communication</p>
             </div>
            <div class='content'>
                <div class='message-content'>
                    " . nl2br(htmlspecialchars($message)) . "
                </div>
            </div>
                         <div class='footer'>
                 <p>This email was sent from the Migration.ng portal.</p>
                 <p>If you have any questions, please contact us at <a href='mailto:services@migration.com.ng'>services@migration.com.ng</a></p>
                 <div style='text-align: center; margin: 20px 0;'>
                     <img src='https://migration.ng/assets/images/m.-preview.png' alt='Migration.ng' style='width: 100px; height: auto;'>
                 </div>
                 <p>&copy; " . date('Y') . " Migration.ng. All rights reserved.</p>
             </div>
        </div>
    </body>
    </html>";
    
    return $html;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Management - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="../index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="../index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <a href="../admin.php" class="text-red-600 hover:text-red-700 text-sm font-medium">
                        <i class="fas fa-cog mr-1"></i>Admin Panel
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($admin['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($admin['firstname'] . ' ' . $admin['lastname']); ?></p>
                            <span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                Admin
                            </span>
                        </div>
                    </div>
                    <a href="../logout.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-sign-out-alt"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Email Management</h1>
                    <p class="text-gray-600 mt-2">Send emails to applicants and track communications</p>
                </div>
                <a href="../admin.php" class="text-gray-600 hover:text-gray-800">
                    <i class="fas fa-arrow-left mr-1"></i>
                    Back to Admin Panel
                </a>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($error)): ?>
            <div class="mb-6 bg-red-50 border border-red-200 rounded-md p-4">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-circle text-red-400"></i>
                    </div>
                    <div class="ml-3">
                        <h3 class="text-sm font-medium text-red-800">Error</h3>
                        <p class="mt-1 text-sm text-red-700"><?php echo htmlspecialchars($error); ?></p>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if (isset($successMessage)): ?>
            <div class="mb-6 bg-green-50 border border-green-200 rounded-md p-4">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-check-circle text-green-400"></i>
                    </div>
                    <div class="ml-3">
                        <h3 class="text-sm font-medium text-green-800">Success!</h3>
                        <p class="mt-1 text-sm text-green-700"><?php echo htmlspecialchars($successMessage); ?></p>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <!-- Email Form -->
            <div class="lg:col-span-2">
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Send Email</h2>
                    </div>
                    <div class="p-6">
                        <form method="POST" id="emailForm">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                            
                            <!-- Recipient Selection -->
                            <div class="mb-6">
                                <label for="recipient_email" class="block text-sm font-medium text-gray-700 mb-2">Recipient Email</label>
                                <div class="flex space-x-2">
                                    <input type="email" name="recipient_email" id="recipient_email" required 
                                           class="flex-1 px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" 
                                           placeholder="Enter email address or select from list below">
                                    <button type="button" onclick="showRecipientModal()" 
                                            class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                        <i class="fas fa-users mr-1"></i>
                                        Select
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Subject -->
                            <div class="mb-6">
                                <label for="subject" class="block text-sm font-medium text-gray-700 mb-2">Subject</label>
                                <input type="text" name="subject" id="subject" required 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" 
                                       placeholder="Enter email subject">
                            </div>
                            
                            <!-- Message -->
                            <div class="mb-6">
                                <label for="message" class="block text-sm font-medium text-gray-700 mb-2">Message</label>
                                <textarea name="message" id="message" rows="10" required 
                                          class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" 
                                          placeholder="Enter your message here..."></textarea>
                                                                 <p class="mt-1 text-sm text-gray-500">You can use plain text formatting. Line breaks will be automatically converted to paragraphs.</p>
                            </div>
                            
                            <!-- Quick Templates -->
                            <div class="mb-6">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Quick Templates</label>
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-2">
                                    <button type="button" onclick="loadTemplate('welcome')" 
                                            class="px-3 py-2 text-sm border border-gray-300 rounded-md hover:bg-gray-50">
                                        Welcome Email
                                    </button>
                                    <button type="button" onclick="loadTemplate('status_update')" 
                                            class="px-3 py-2 text-sm border border-gray-300 rounded-md hover:bg-gray-50">
                                        Status Update
                                    </button>
                                    <button type="button" onclick="loadTemplate('document_request')" 
                                            class="px-3 py-2 text-sm border border-gray-300 rounded-md hover:bg-gray-50">
                                        Document Request
                                    </button>
                                                                         <button type="button" onclick="loadTemplate('payment_reminder')" 
                                             class="px-3 py-2 text-sm border border-gray-300 rounded-md hover:bg-gray-50">
                                         Payment Reminder
                                     </button>
                                     <button type="button" onclick="loadTemplate('delivery_request')" 
                                             class="px-3 py-2 text-sm border border-gray-300 rounded-md hover:bg-gray-50">
                                         Delivery Request
                                     </button>
                                </div>
                            </div>
                            
                            <!-- Send Button -->
                            <div class="flex justify-end">
                                <button type="submit" name="send_email" 
                                        class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <i class="fas fa-paper-plane mr-2"></i>
                                    Send Email
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="space-y-6">
                <!-- Statistics -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Email Statistics</h3>
                    </div>
                    <div class="p-6">
                        <dl class="space-y-4">
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Total Applicants</dt>
                                <dd class="mt-1 text-2xl font-bold text-gray-900"><?php echo count($applicants); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Active Applications</dt>
                                <dd class="mt-1 text-2xl font-bold text-gray-900">
                                    <?php echo array_sum(array_column($applicants, 'application_count')); ?>
                                </dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Document Requests</dt>
                                <dd class="mt-1 text-2xl font-bold text-gray-900">
                                    <?php echo array_sum(array_column($applicants, 'document_request_count')); ?>
                                </dd>
                            </div>
                        </dl>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Quick Actions</h3>
                    </div>
                    <div class="p-6 space-y-3">
                        <a href="../admin-applications.php" class="w-full inline-flex justify-center items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                            <i class="fas fa-list mr-2"></i>
                            View Applications
                        </a>
                        <a href="../admin.php" class="w-full inline-flex justify-center items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                            <i class="fas fa-cog mr-2"></i>
                            Admin Panel
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Recipient Selection Modal -->
    <div id="recipientModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-2xl w-full">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 class="text-lg font-medium text-gray-900">Select Recipient</h3>
                </div>
                <div class="px-6 py-4 max-h-96 overflow-y-auto">
                    <div class="space-y-2">
                        <?php foreach ($applicants as $applicant): ?>
                            <div class="flex items-center justify-between p-3 border border-gray-200 rounded-lg hover:bg-gray-50 cursor-pointer" 
                                 onclick="selectRecipient('<?php echo htmlspecialchars($applicant['email']); ?>', '<?php echo htmlspecialchars($applicant['firstname'] . ' ' . $applicant['lastname']); ?>')">
                                <div>
                                    <p class="font-medium text-gray-900"><?php echo htmlspecialchars($applicant['firstname'] . ' ' . $applicant['lastname']); ?></p>
                                    <p class="text-sm text-gray-500"><?php echo htmlspecialchars($applicant['email']); ?></p>
                                    <p class="text-xs text-gray-400">
                                        Applications: <?php echo $applicant['application_count']; ?> | 
                                        Documents: <?php echo $applicant['document_request_count']; ?>
                                    </p>
                                </div>
                                <i class="fas fa-chevron-right text-gray-400"></i>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <div class="px-6 py-4 bg-gray-50 flex justify-end">
                    <button type="button" onclick="closeRecipientModal()" class="px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50">
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script>
        function showRecipientModal() {
            document.getElementById('recipientModal').classList.remove('hidden');
        }
        
        function closeRecipientModal() {
            document.getElementById('recipientModal').classList.add('hidden');
        }
        
        function selectRecipient(email, name) {
            document.getElementById('recipient_email').value = email;
            closeRecipientModal();
        }
        
                 function loadTemplate(type) {
             const templates = {
                 welcome: {
                     subject: 'Welcome to Migration.ng - Your Application Status',
                     message: `Dear [Applicant Name],

Welcome to Migration.ng! We're excited to help you with your migration journey.

Your application has been received and is currently being processed. We'll keep you updated on any developments.

If you have any questions, please don't hesitate to contact us.

Best regards,
The Migration.ng Team`
                 },
                 status_update: {
                     subject: 'Application Status Update - Migration.ng',
                     message: `Dear [Applicant Name],

We have an important update regarding your application.

Current Status: [Status]
Update Details: [Details]

Please log in to your portal to view the complete details and any required actions.

Best regards,
The Migration.ng Team`
                 },
                 document_request: {
                     subject: 'Document Request - Migration.ng',
                     message: `Dear [Applicant Name],

We need additional documents to process your application.

Required Documents:
• [Document 1]
• [Document 2]

Please upload these documents through your portal or contact us if you have any questions.

Best regards,
The Migration.ng Team`
                 },
                                   payment_reminder: {
                      subject: 'Payment Reminder - Migration.ng',
                      message: `Dear [Applicant Name],

This is a friendly reminder about your pending payment.

Amount Due: [Amount]
Due Date: [Date]

Please complete your payment to avoid delays in processing your application.

Best regards,
The Migration.ng Team`
                  },
                  delivery_request: {
                      subject: 'Document Delivery Request - Migration.ng',
                      message: `Dear [Applicant Name],

Your document(s) have been processed and are ready for delivery.

Document(s) Ready:
• [Document Type 1]
• [Document Type 2]

Please let us know your preferred delivery method:

1. **Courier Delivery** - We can arrange for secure courier delivery to your address
2. **Pickup** - You can collect from our office
3. **Digital Copy** - We can send you a digital copy via email

Please reply to this email with your preferred delivery method and any additional details (address for courier, preferred pickup time, etc.).

Best regards,
The Migration.ng Team`
                  }
             };
            
            if (templates[type]) {
                document.getElementById('subject').value = templates[type].subject;
                document.getElementById('message').value = templates[type].message;
            }
        }
        
        // Auto-hide success messages after 5 seconds
        setTimeout(function() {
            const successMessages = document.querySelectorAll('.bg-green-50');
            successMessages.forEach(function(message) {
                message.style.display = 'none';
            });
        }, 5000);
    </script>
</body>
</html>
