<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get user data
try {
    $pdo = getDatabaseConnection();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    // Get user's applications from portal_applications table
    $stmt = $pdo->prepare("SELECT * FROM portal_applications WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$_SESSION['user_id']]);
    $applications = $stmt->fetchAll();
    
    // Get document requests
    $stmt = $pdo->prepare("SELECT * FROM document_requests WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$_SESSION['user_id']]);
    $documentRequests = $stmt->fetchAll();
    
    // Combine and sort all recent items
    $allRecentItems = [];
    
    // Add applications
    foreach ($applications as $app) {
        $allRecentItems[] = [
            'id' => $app['id'],
            'type' => 'application',
            'title' => html_entity_decode($app['degree']),
            'subtitle' => $app['university'],
            'country' => $app['country'],
            'status' => $app['status'],
            'created_at' => $app['created_at']
        ];
    }
    
    // Add document requests
    foreach ($documentRequests as $doc) {
        $allRecentItems[] = [
            'id' => $doc['id'],
            'type' => 'document_request',
            'title' => $doc['document_type'],
            'subtitle' => $doc['destination_country'],
            'country' => $doc['destination_country'],
            'status' => $doc['status'],
            'created_at' => $doc['created_at']
        ];
    }
    
    // Sort by creation date (newest first)
    usort($allRecentItems, function($a, $b) {
        return strtotime($b['created_at']) - strtotime($a['created_at']);
    });
    
    // Get application statistics
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
        SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress,
        SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
        SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
        FROM portal_applications WHERE user_id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $stats = $stmt->fetch();
    
    // Check for recent updates (last 24 hours)
    $stmt = $pdo->prepare("
        SELECT 
            'document_request' as type,
            dr.id,
            dr.request_number,
            dr.document_type,
            dru.created_at,
            dru.update_text
        FROM document_request_updates dru
        JOIN document_requests dr ON dru.request_id = dr.id
        WHERE dr.user_id = ? 
        AND dru.update_type = 'admin'
        AND dru.created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        UNION ALL
        SELECT 
            'application' as type,
            pa.id,
            CONCAT('APP-', pa.id) as request_number,
            pa.degree,
            au.created_at,
            au.update_text
        FROM application_updates au
        JOIN portal_applications pa ON au.application_id = pa.id
        WHERE pa.user_id = ? 
        AND au.update_type = 'admin'
        AND au.created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ORDER BY created_at DESC
    ");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $recentUpdates = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Portal error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Applications - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="/" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($user['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></p>
                            <p class="text-xs text-gray-500"><?php echo htmlspecialchars($user['email']); ?></p>
                            <?php if (UserAuth::isAdmin()): ?>
                                <span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                    Admin
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php if (UserAuth::isAdmin()): ?>
                        <a href="admin.php" class="text-red-600 hover:text-red-700 text-sm font-medium">
                            <i class="fas fa-cog mr-1"></i>Admin Panel
                        </a>
                    <?php endif; ?>
                    <a href="logout.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-sign-out-alt"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Welcome Section -->
        <div class="mb-8">
            <h1 class="text-3xl font-bold text-gray-900">Welcome back, <?php echo htmlspecialchars($user['firstname']); ?>!</h1>
            <p class="text-gray-600 mt-2">Track your applications and manage your migration journey.</p>
        </div>

        <!-- New Updates Alert -->
        <?php if (!empty($recentUpdates)): ?>
            <div class="mb-8 bg-blue-50 border border-blue-200 rounded-lg p-4">
                <div class="flex items-start">
                    <div class="flex-shrink-0">
                        <i class="fas fa-bell text-blue-600 text-lg"></i>
                    </div>
                    <div class="ml-3 flex-1">
                        <h3 class="text-sm font-medium text-blue-800">
                            You have <?php echo count($recentUpdates); ?> new update<?php echo count($recentUpdates) > 1 ? 's' : ''; ?>!
                        </h3>
                                                 <div class="mt-2 text-sm text-blue-700">
                             <?php foreach (array_slice($recentUpdates, 0, 3) as $update): ?>
                                 <div class="mb-2">
                                     <span class="font-medium">
                                         <?php echo $update['type'] === 'document_request' ? 'Document Request' : 'Application'; ?> 
                                         #<?php echo htmlspecialchars($update['request_number']); ?>:
                                     </span>
                                     <span class="text-blue-700">
                                         <?php echo htmlspecialchars(substr($update['update_text'], 0, 100)); ?>
                                         <?php if (strlen($update['update_text']) > 100): ?>...<?php endif; ?>
                                     </span>
                                     <a href="<?php echo $update['type'] === 'document_request' ? 'document-request-details.php?id=' . $update['id'] : 'application-details.php?id=' . $update['id']; ?>#updates" class="text-blue-600 hover:text-blue-800 hover:underline ml-2 font-medium">
                                         View Update →
                                     </a>
                                                                           <span class="text-blue-500 text-xs ml-2">
                                          <?php echo date('M j, g:i A', strtotime($update['created_at'])); ?> GMT
                                      </span>
                                 </div>
                             <?php endforeach; ?>
                             <?php if (count($recentUpdates) > 3): ?>
                                 <div class="text-blue-600 text-xs">
                                     +<?php echo count($recentUpdates) - 3; ?> more update<?php echo (count($recentUpdates) - 3) > 1 ? 's' : ''; ?>
                                 </div>
                             <?php endif; ?>
                         </div>
                    </div>
                    <div class="ml-4 flex-shrink-0">
                        <button type="button" class="text-blue-400 hover:text-blue-600" onclick="this.parentElement.parentElement.parentElement.style.display='none'">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 mb-8">
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-blue-100 rounded-lg">
                        <i class="fas fa-file-alt text-blue-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Total Applications</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['total'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-yellow-100 rounded-lg">
                        <i class="fas fa-clock text-yellow-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Pending</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['pending'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-blue-100 rounded-lg">
                        <i class="fas fa-spinner text-blue-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">In Progress</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['in_progress'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-green-100 rounded-lg">
                        <i class="fas fa-check-circle text-green-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Completed</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['completed'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white rounded-lg shadow p-6">
                <div class="flex items-center">
                    <div class="p-2 bg-red-100 rounded-lg">
                        <i class="fas fa-times-circle text-red-600"></i>
                    </div>
                    <div class="ml-4">
                        <p class="text-sm font-medium text-gray-600">Rejected</p>
                        <p class="text-2xl font-bold text-gray-900"><?php echo $stats['rejected'] ?? 0; ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="bg-white rounded-lg shadow mb-8">
            <div class="px-6 py-4 border-b border-gray-200">
                <h2 class="text-lg font-medium text-gray-900">Quick Actions</h2>
            </div>
            <div class="p-6">
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <a href="new-application.php" class="flex items-center p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                        <div class="p-2 bg-blue-100 rounded-lg">
                            <i class="fas fa-plus text-blue-600"></i>
                        </div>
                        <div class="ml-4">
                            <h3 class="font-medium text-gray-900">New Application</h3>
                            <p class="text-sm text-gray-500">Submit a new university application</p>
                        </div>
                    </a>
                    
                    <a href="documents.php" class="flex items-center p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                        <div class="p-2 bg-green-100 rounded-lg">
                            <i class="fas fa-upload text-green-600"></i>
                        </div>
                        <div class="ml-4">
                            <h3 class="font-medium text-gray-900">Upload Documents</h3>
                            <p class="text-sm text-gray-500">Upload required documents</p>
                        </div>
                    </a>
                    
                    <a href="document-services.php" class="flex items-center p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                        <div class="p-2 bg-purple-100 rounded-lg">
                            <i class="fas fa-stamp text-purple-600"></i>
                        </div>
                        <div class="ml-4">
                            <h3 class="font-medium text-gray-900">Document Services</h3>
                            <p class="text-sm text-gray-500">Legalize and authenticate documents</p>
                        </div>
                    </a>
                    
                    <a href="profile.php" class="flex items-center p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                        <div class="p-2 bg-purple-100 rounded-lg">
                            <i class="fas fa-user text-purple-600"></i>
                        </div>
                        <div class="ml-4">
                            <h3 class="font-medium text-gray-900">Update Profile</h3>
                            <p class="text-sm text-gray-500">Manage your account details</p>
                        </div>
                    </a>
                    
                    <a href="invoices.php" class="flex items-center p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                        <div class="p-2 bg-orange-100 rounded-lg">
                            <i class="fas fa-file-invoice text-orange-600"></i>
                        </div>
                        <div class="ml-4">
                            <h3 class="font-medium text-gray-900">Invoices</h3>
                            <p class="text-sm text-gray-500">View and manage your invoices</p>
                        </div>
                    </a>
                </div>
            </div>
        </div>

        <!-- Recent Applications & Document Requests -->
        <div class="bg-white rounded-lg shadow">
            <div class="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
                <h2 class="text-lg font-medium text-gray-900">Recent Applications & Document Requests</h2>
                <div class="flex space-x-4">
                    <a href="applications.php" class="text-blue-600 hover:text-blue-800 text-sm font-medium">View Applications</a>
                    <a href="document-services.php" class="text-blue-600 hover:text-blue-800 text-sm font-medium">View Documents</a>
                </div>
            </div>
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Title</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Details</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Country</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($allRecentItems)): ?>
                            <tr>
                                <td colspan="7" class="px-6 py-4 text-center text-gray-500">
                                    No applications or document requests found. 
                                    <div class="mt-2 space-x-4">
                                        <a href="new-application.php" class="text-blue-600 hover:text-blue-800">Start your first application</a>
                                        <span class="text-gray-400">|</span>
                                        <a href="document-services.php" class="text-blue-600 hover:text-blue-800">Request document services</a>
                                    </div>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach (array_slice($allRecentItems, 0, 5) as $item): ?>
                                <tr>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="w-8 h-8 rounded-full <?php echo $item['type'] === 'document_request' ? 'bg-purple-100' : 'bg-blue-100'; ?> flex items-center justify-center mr-3">
                                                <i class="<?php echo $item['type'] === 'document_request' ? 'fas fa-stamp text-purple-600' : 'fas fa-graduation-cap text-blue-600'; ?> text-sm"></i>
                                            </div>
                                            <span class="text-sm font-medium text-gray-900">
                                                <?php echo $item['type'] === 'document_request' ? 'Document' : 'Application'; ?>
                                            </span>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($item['title']); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-900"><?php echo htmlspecialchars($item['subtitle']); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-900"><?php echo htmlspecialchars($item['country']); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php
                                        $status = $item['status'] ?? 'pending';
                                        $statusClasses = [
                                            'pending' => 'bg-yellow-100 text-yellow-800',
                                            'submitted' => 'bg-blue-100 text-blue-800',
                                            'in_progress' => 'bg-blue-100 text-blue-800',
                                            'completed' => 'bg-green-100 text-green-800',
                                            'rejected' => 'bg-red-100 text-red-800',
                                            'cancelled' => 'bg-gray-100 text-gray-800'
                                        ];
                                        $statusText = [
                                            'pending' => 'Pending',
                                            'submitted' => 'Submitted',
                                            'in_progress' => 'In Progress',
                                            'completed' => 'Completed',
                                            'rejected' => 'Rejected',
                                            'cancelled' => 'Cancelled'
                                        ];
                                        ?>
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?php echo $statusClasses[$status]; ?>">
                                            <?php echo $statusText[$status]; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo date('M j, Y', strtotime($item['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <?php if ($item['type'] === 'document_request'): ?>
                                            <a href="document-request-details.php?id=<?php echo $item['id']; ?>" class="text-blue-600 hover:text-blue-900 mr-3">View</a>
                                        <?php else: ?>
                                            <a href="application-details.php?id=<?php echo $item['id']; ?>" class="text-blue-600 hover:text-blue-900 mr-3">View</a>
                                            <a href="edit-application.php?id=<?php echo $item['id']; ?>" class="text-indigo-600 hover:text-indigo-900">Edit</a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script>
        // Auto-refresh dashboard every 30 seconds
        setTimeout(function() {
            location.reload();
        }, 30000);
    </script>
</body>
</html> 