<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get user data and invoices
try {
    $pdo = getDatabaseConnection();
    
    // Get user data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    // Get filter parameter
    $filter = $_GET['filter'] ?? 'all';
    
    // Build the WHERE clause based on filter
    $whereClause = "WHERE dr.user_id = ?";
    if ($filter === 'pending') {
        $whereClause .= " AND dr.payment_status = 'pending'";
    } elseif ($filter === 'paid') {
        $whereClause .= " AND dr.payment_status = 'paid'";
    } elseif ($filter === 'completed') {
        $whereClause .= " AND dr.status = 'completed'";
    } elseif ($filter === 'cancelled') {
        $whereClause .= " AND dr.status = 'cancelled'";
    }
    
    // Get document request invoices
    $stmt = $pdo->prepare("
        SELECT 
            dr.id,
            dr.request_number,
            dr.document_type,
            dr.total_amount,
            dr.paid_amount,
            dr.payment_status,
            dr.status,
            dr.created_at,
            'document_request' as type,
            dst.name as service_name
        FROM document_requests dr
        LEFT JOIN document_service_types dst ON dr.service_type_id = dst.id
        $whereClause
        ORDER BY dr.created_at DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $documentInvoices = $stmt->fetchAll();
    
    // For now, only show document request invoices since application pricing isn't implemented yet
    $allInvoices = $documentInvoices;
    
} catch (Exception $e) {
    error_log("Invoices error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Get status color
function getPaymentStatusColor($status) {
    switch ($status) {
        case 'pending': return 'bg-yellow-100 text-yellow-800';
        case 'partial': return 'bg-orange-100 text-orange-800';
        case 'paid': return 'bg-green-100 text-green-800';
        case 'overdue': return 'bg-red-100 text-red-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}

// Get type icon (only document requests for now)
function getTypeIcon($type) {
    return 'fas fa-stamp text-purple-600';
}

// Get type color (only document requests for now)
function getTypeColor($type) {
    return 'bg-purple-100';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoices - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($user['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Invoices</h1>
                    <p class="text-gray-600 mt-2">View and manage your pending invoices</p>
                </div>
                <a href="index.php" class="text-gray-600 hover:text-gray-800">
                    <i class="fas fa-arrow-left mr-1"></i>
                    Back to Dashboard
                </a>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (isset($error)): ?>
            <?php ErrorHandler::displayValidationError($error); ?>
        <?php endif; ?>

        <!-- Filter Tabs -->
        <div class="bg-white rounded-lg shadow mb-6">
            <div class="px-6 py-4">
                <div class="flex space-x-8">
                    <a href="?filter=all" class="px-3 py-2 text-sm font-medium rounded-md <?php echo $filter === 'all' ? 'bg-blue-100 text-blue-700' : 'text-gray-500 hover:text-gray-700'; ?>">
                        All Invoices
                    </a>
                    <a href="?filter=pending" class="px-3 py-2 text-sm font-medium rounded-md <?php echo $filter === 'pending' ? 'bg-yellow-100 text-yellow-700' : 'text-gray-500 hover:text-gray-700'; ?>">
                        Pending Payment
                    </a>
                    <a href="?filter=paid" class="px-3 py-2 text-sm font-medium rounded-md <?php echo $filter === 'paid' ? 'bg-green-100 text-green-700' : 'text-gray-500 hover:text-gray-700'; ?>">
                        Paid
                    </a>
                    <a href="?filter=completed" class="px-3 py-2 text-sm font-medium rounded-md <?php echo $filter === 'completed' ? 'bg-green-100 text-green-700' : 'text-gray-500 hover:text-gray-700'; ?>">
                        Completed
                    </a>
                    <a href="?filter=cancelled" class="px-3 py-2 text-sm font-medium rounded-md <?php echo $filter === 'cancelled' ? 'bg-red-100 text-red-700' : 'text-gray-500 hover:text-gray-700'; ?>">
                        Cancelled
                    </a>
                </div>
            </div>
        </div>

        <!-- Invoices List -->
        <div class="bg-white rounded-lg shadow">
            <div class="px-6 py-4 border-b border-gray-200">
                <h2 class="text-lg font-medium text-gray-900">
                    <?php 
                    switch($filter) {
                        case 'pending': echo 'Pending Payment Invoices'; break;
                        case 'paid': echo 'Paid Invoices'; break;
                        case 'completed': echo 'Completed Invoices'; break;
                        case 'cancelled': echo 'Cancelled Invoices'; break;
                        default: echo 'All Invoices'; break;
                    }
                    ?>
                </h2>
            </div>
            
            <?php if (empty($allInvoices)): ?>
                <div class="px-6 py-12 text-center">
                    <div class="mx-auto w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4">
                        <i class="fas fa-file-invoice text-gray-400 text-2xl"></i>
                    </div>
                                         <h3 class="text-lg font-medium text-gray-900 mb-2">
                         <?php 
                         switch($filter) {
                             case 'pending': echo 'No pending payment invoices'; break;
                             case 'paid': echo 'No paid invoices'; break;
                             case 'completed': echo 'No completed invoices'; break;
                             case 'cancelled': echo 'No cancelled invoices'; break;
                             default: echo 'No invoices found'; break;
                         }
                         ?>
                     </h3>
                     <p class="text-gray-500 mb-6">
                         <?php 
                         switch($filter) {
                             case 'pending': echo 'You don\'t have any pending payment invoices at the moment.'; break;
                             case 'paid': echo 'You don\'t have any paid invoices yet.'; break;
                             case 'completed': echo 'You don\'t have any completed invoices yet.'; break;
                             case 'cancelled': echo 'You don\'t have any cancelled invoices.'; break;
                             default: echo 'You don\'t have any invoices at the moment.'; break;
                         }
                         ?>
                     </p>
                    <div class="space-x-4">
                        <a href="new-application.php" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700">
                            <i class="fas fa-plus mr-2"></i>
                            New Application
                        </a>
                        <a href="document-services.php" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                            <i class="fas fa-stamp mr-2"></i>
                            Document Services
                        </a>
                    </div>
                </div>
            <?php else: ?>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Invoice</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Description</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($allInvoices as $invoice): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="flex-shrink-0 h-10 w-10">
                                                <div class="h-10 w-10 rounded-full <?php echo getTypeColor($invoice['type']); ?> flex items-center justify-center">
                                                    <i class="<?php echo getTypeIcon($invoice['type']); ?>"></i>
                                                </div>
                                            </div>
                                            <div class="ml-4">
                                                                                             <div class="text-sm font-medium text-gray-900">
                                                 <?php echo htmlspecialchars($invoice['request_number']); ?>
                                             </div>
                                             <div class="text-sm text-gray-500">
                                                 Document Request
                                             </div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-gray-800">
                                            Document Request
                                        </span>
                                    </td>
                                    <td class="px-6 py-4">
                                        <div class="text-sm text-gray-900">
                                            <?php echo htmlspecialchars($invoice['document_type']); ?>
                                            <?php if ($invoice['service_name']): ?>
                                                <br><span class="text-gray-500"><?php echo htmlspecialchars($invoice['service_name']); ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900">
                                            ₦<?php echo number_format($invoice['total_amount'], 2); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="space-y-1">
                                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo getPaymentStatusColor($invoice['payment_status']); ?>">
                                                <?php echo ucfirst($invoice['payment_status']); ?>
                                            </span>
                                            <div class="text-xs text-gray-500">
                                                Status: <?php echo ucfirst(str_replace('_', ' ', $invoice['status'])); ?>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo date('M j, Y', strtotime($invoice['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <a href="document-request-details.php?id=<?php echo $invoice['id']; ?>" class="text-blue-600 hover:text-blue-900 mr-3">View Details</a>
                                        <?php if ($invoice['payment_status'] === 'pending' && $invoice['status'] === 'pending'): ?>
                                            <a href="submit-order.php?request_id=<?php echo $invoice['id']; ?>" class="text-green-600 hover:text-green-900">Submit Order</a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>

        <!-- Information Section -->
        <div class="mt-8 grid grid-cols-1 lg:grid-cols-2 gap-8">
            <!-- How Invoices Work -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-medium text-gray-900 mb-4">How Our Invoice System Works</h3>
                <div class="space-y-4">
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <span class="text-blue-600 font-medium text-sm">1</span>
                            </div>
                        </div>
                        <div class="ml-4">
                            <h4 class="text-sm font-medium text-gray-900">Submit Order</h4>
                            <p class="text-sm text-gray-500">Submit your application or document request for review</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <span class="text-blue-600 font-medium text-sm">2</span>
                            </div>
                        </div>
                        <div class="ml-4">
                            <h4 class="text-sm font-medium text-gray-900">Review & Quote</h4>
                            <p class="text-sm text-gray-500">We review your request and provide an exact cost quote</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <span class="text-blue-600 font-medium text-sm">3</span>
                            </div>
                        </div>
                        <div class="ml-4">
                            <h4 class="text-sm font-medium text-gray-900">Receive Invoice</h4>
                            <p class="text-sm text-gray-500">You'll receive a detailed invoice with exact pricing</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                                <span class="text-blue-600 font-medium text-sm">4</span>
                            </div>
                        </div>
                        <div class="ml-4">
                            <h4 class="text-sm font-medium text-gray-900">Make Payment</h4>
                            <p class="text-sm text-gray-500">Pay the invoice and we'll begin processing your request</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Payment Methods -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Accepted Payment Methods</h3>
                <div class="space-y-4">
                    <div class="flex items-center">
                        <i class="fas fa-university text-green-500 mr-3"></i>
                        <div>
                            <h4 class="text-sm font-medium text-gray-900">Bank Transfer</h4>
                            <p class="text-sm text-gray-500">Direct bank transfer to our account</p>
                        </div>
                    </div>
                    <div class="flex items-center">
                        <i class="fas fa-credit-card text-blue-500 mr-3"></i>
                        <div>
                            <h4 class="text-sm font-medium text-gray-900">Card Payment</h4>
                            <p class="text-sm text-gray-500">Secure online card payment</p>
                        </div>
                    </div>
                    <div class="flex items-center">
                        <i class="fas fa-money-bill-wave text-green-500 mr-3"></i>
                        <div>
                            <h4 class="text-sm font-medium text-gray-900">Cash Payment</h4>
                            <p class="text-sm text-gray-500">Cash payment at our office</p>
                        </div>
                    </div>
                </div>
                
                <div class="mt-6 p-4 bg-blue-50 rounded-lg">
                    <h4 class="text-sm font-medium text-blue-900 mb-2">Need Help?</h4>
                    <p class="text-sm text-blue-700 mb-3">If you have questions about your invoice or payment, contact our support team.</p>
                    <a href="mailto:services@migration.ng" class="text-blue-600 hover:text-blue-800 text-sm font-medium">
                        <i class="fas fa-envelope mr-1"></i>
                        Contact Support
                    </a>
                </div>
            </div>
        </div>
    </div>
</body>
</html> 