<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../config/whatsapp.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';
require_once '../includes/EmailVerification.php';

// Initialize error handling
ErrorHandler::init();

// Initialize variables
$errors = [];
$success = false;

// Handle login form submission
if (isset($_POST['login'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        // Validate input
        $email = Validator::validateEmail($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        
        if (!$email) {
            $errors[] = "Please enter a valid email address";
        }
        if (empty($password)) {
            $errors[] = "Please enter your password";
        }
        
        // If no validation errors, attempt login
        if (empty($errors)) {
            $pdo = getDatabaseConnection();
            $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch();
            
            if ($user && password_verify($password, $user['password'])) {
                // Check if email is verified
                if (!$user['email_verified']) {
                    $errors[] = "Please verify your email address before logging in. Check your inbox for a verification link, or <a href='verify-email.php' class='text-blue-600 hover:text-blue-500'>click here to resend the verification email</a>.";
                } else {
                    // Login successful
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['user_email'] = $user['email'];
                    $_SESSION['user_role'] = $user['role']; // Store user role
                    
                    // Update last login
                    $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                    $stmt->execute([$user['id']]);
                    
                    header('Location: index.php');
                    exit();
                }
            } else {
                $errors[] = "Invalid email or password";
            }
        }
    } catch (Exception $e) {
        error_log("Login error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Handle registration form submission
if (isset($_POST['register'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        // Validate input
        $firstName = Validator::validateText($_POST['firstname'] ?? '', 50);
        $lastName = Validator::validateText($_POST['lastname'] ?? '', 50);
        $email = Validator::validateEmail($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        
        if (!$firstName) $errors[] = "Please enter your first name";
        if (!$lastName) $errors[] = "Please enter your last name";
        if (!$email) $errors[] = "Please enter a valid email address";
        if (strlen($password) < 8) $errors[] = "Password must be at least 8 characters long";
        if ($password !== $confirmPassword) $errors[] = "Passwords do not match";
        
        // If no validation errors, attempt registration
        if (empty($errors)) {
            $pdo = getDatabaseConnection();
            
            // Check if email already exists
            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            if ($stmt->fetch()) {
                $errors[] = "An account with this email already exists";
            } else {
                // Create new user
                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("INSERT INTO users (firstname, lastname, email, password, role, created_at) VALUES (?, ?, ?, ?, 'user', NOW())");
                $stmt->execute([$firstName, $lastName, $email, $hashedPassword]);
                
                $userId = $pdo->lastInsertId();
                
                // Create email verification
                $verificationToken = EmailVerification::createVerification($userId, $email);
                
                if ($verificationToken) {
                    // Send verification email
                    $userData = [
                        'id' => $userId,
                        'firstname' => $firstName,
                        'lastname' => $lastName,
                        'email' => $email
                    ];
                    
                    if (EmailService::sendEmailVerification($userData, $verificationToken)) {
                        $success = true;
                        $successMessage = "Account created successfully! Please check your email and click the verification link to activate your account.";
                    } else {
                        // If email sending fails, still create the account but inform user
                        $success = true;
                        $successMessage = "Account created successfully! However, we couldn't send the verification email. Please contact support to verify your account.";
                    }
                } else {
                    // If verification creation fails, still create the account but inform user
                    $success = true;
                    $successMessage = "Account created successfully! However, there was an issue with email verification. Please contact support to verify your account.";
                }
            }
        }
    } catch (Exception $e) {
        error_log("Registration error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Migration.ng Portal</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="../assets/favicon/favicon.ico">
    <link rel="icon" type="image/png" sizes="16x16" href="../assets/favicon/favicon-16x16.png">
    <link rel="icon" type="image/png" sizes="32x32" href="../assets/favicon/favicon-32x32.png">
    <link rel="apple-touch-icon" sizes="180x180" href="../assets/favicon/apple-touch-icon.png">
    <link rel="manifest" href="../assets/favicon/site.webmanifest">
    
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50 min-h-screen flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
    <div class="max-w-md w-full space-y-8">
        <!-- Logo and Title -->
        <div class="text-center">
            <a href="/" class="flex justify-center">
                <img src="/assets/images/m..png" alt="Migration.ng" class="h-12 w-auto">
            </a>
            <h2 class="mt-6 text-3xl font-extrabold text-gray-900">Welcome to your portal</h2>
            <p class="mt-2 text-sm text-gray-600">
                Track your applications and manage your migration journey
            </p>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if ($success): ?>
            <?php ErrorHandler::displaySuccess($successMessage); ?>
        <?php endif; ?>

        <!-- Login Form -->
        <div class="bg-white py-8 px-6 shadow rounded-lg sm:px-10" id="loginContainer">
            <form method="POST" action="" id="loginForm">
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                
                <div class="mb-6">
                    <label for="email" class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                    <input
                        type="email"
                        name="email"
                        id="email"
                        required
                        class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Enter your email"
                    />
                </div>

                <div class="mb-6">
                    <label for="password" class="block text-sm font-medium text-gray-700 mb-2">Password</label>
                    <input
                        type="password"
                        name="password"
                        id="password"
                        required
                        class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Enter your password"
                    />
                </div>

                <div class="flex items-center justify-between mb-6">
                    <div class="flex items-center">
                        <input
                            type="checkbox"
                            id="remember"
                            name="remember"
                            class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                        />
                        <label for="remember" class="ml-2 block text-sm text-gray-900">
                            Remember me
                        </label>
                    </div>
                    <a href="forgot-password.php" class="text-sm text-blue-600 hover:text-blue-500">
                        Forgot password?
                    </a>
                </div>

                <div>
                    <button
                        type="submit"
                        name="login"
                        class="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                    >
                        Sign In
                    </button>
                </div>
            </form>

            <!-- Email Verification Help -->
            <div class="mt-4 text-center">
                <p class="text-sm text-gray-600">
                    Need to verify your email? 
                    <a href="verify-email.php" class="text-blue-600 hover:text-blue-500 font-medium">
                        Resend verification email
                    </a>
                </p>
            </div>

            <div class="mt-6">
                <div class="relative">
                    <div class="absolute inset-0 flex items-center">
                        <div class="w-full border-t border-gray-300"></div>
                    </div>
                    <div class="relative flex justify-center text-sm">
                        <span class="px-2 bg-white text-gray-500">Don't have an account?</span>
                    </div>
                </div>

                <div class="mt-6">
                    <button
                        type="button"
                        onclick="toggleForms()"
                        class="w-full flex justify-center py-2 px-4 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                    >
                        Create Account
                    </button>
                </div>
            </div>
        </div>

        <!-- Registration Form (Hidden by default) -->
        <div class="bg-white py-8 px-6 shadow rounded-lg sm:px-10 hidden" id="registerForm">
            <form method="POST" action="">
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                
                <div class="grid grid-cols-2 gap-4 mb-6">
                    <div>
                        <label for="firstname" class="block text-sm font-medium text-gray-700 mb-2">First Name</label>
                        <input
                            type="text"
                            name="firstname"
                            id="firstname"
                            required
                            class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                            placeholder="First name"
                        />
                    </div>
                    <div>
                        <label for="lastname" class="block text-sm font-medium text-gray-700 mb-2">Last Name</label>
                        <input
                            type="text"
                            name="lastname"
                            id="lastname"
                            required
                            class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Last name"
                        />
                    </div>
                </div>

                <div class="mb-6">
                    <label for="reg_email" class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                    <input
                        type="email"
                        name="email"
                        id="reg_email"
                        required
                        class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Enter your email"
                    />
                </div>

                <div class="mb-6">
                    <label for="reg_password" class="block text-sm font-medium text-gray-700 mb-2">Password</label>
                    <input
                        type="password"
                        name="password"
                        id="reg_password"
                        required
                        class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Create a password"
                    />
                    <p class="mt-1 text-xs text-gray-500">Must be at least 8 characters long</p>
                </div>

                <div class="mb-6">
                    <label for="confirm_password" class="block text-sm font-medium text-gray-700 mb-2">Confirm Password</label>
                    <input
                        type="password"
                        name="confirm_password"
                        id="confirm_password"
                        required
                        class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Confirm your password"
                    />
                </div>

                <div>
                    <button
                        type="submit"
                        name="register"
                        class="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                    >
                        Create Account
                    </button>
                </div>
            </form>

            <div class="mt-6">
                <div class="relative">
                    <div class="absolute inset-0 flex items-center">
                        <div class="w-full border-t border-gray-300"></div>
                    </div>
                    <div class="relative flex justify-center text-sm">
                        <span class="px-2 bg-white text-gray-500">Already have an account?</span>
                    </div>
                </div>

                <div class="mt-6">
                    <button
                        type="button"
                        onclick="toggleForms()"
                        class="w-full flex justify-center py-2 px-4 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                    >
                        Sign In Instead
                    </button>
                </div>
            </div>
        </div>

        <!-- Back to main site -->
        <div class="text-center">
            <a href="/" class="text-sm text-gray-600 hover:text-gray-900">
                <i class="fas fa-arrow-left mr-1"></i>
                Back to main site
            </a>
        </div>
    </div>

    <script>
        function toggleForms() {
            const loginContainer = document.getElementById('loginContainer');
            const registerForm = document.getElementById('registerForm');
            
            if (registerForm.classList.contains('hidden')) {
                // Show register form, hide login form
                loginContainer.style.display = 'none';
                registerForm.classList.remove('hidden');
            } else {
                // Show login form, hide register form
                loginContainer.style.display = 'block';
                registerForm.classList.add('hidden');
            }
        }
    </script>
    
    <!-- Floating WhatsApp Button -->
    <a href="<?php echo WHATSAPP_LINK; ?>" target="_blank" class="fixed bottom-6 right-6 z-50 bg-green-500 hover:bg-green-600 text-white rounded-full p-4 shadow-lg transition-all duration-300 transform hover:scale-110 hover:shadow-xl">
        <i class="fab fa-whatsapp text-2xl"></i>
    </a>
</body>
</html> 