<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
UserAuth::requireLogin();

// Get user data
$currentUser = UserAuth::getCurrentUser();

// Initialize variables
$errors = [];
$success = false;

// Get service types and offices
try {
    $pdo = getDatabaseConnection();
    
    // Get active service types
    $stmt = $pdo->query("SELECT * FROM document_service_types WHERE is_active = 1 ORDER BY name");
    $serviceTypes = $stmt->fetchAll();
    
    // Get active offices
    $stmt = $pdo->query("SELECT * FROM official_offices WHERE is_active = 1 ORDER BY name");
    $offices = $stmt->fetchAll();
    
    // Get pricing data
    $stmt = $pdo->query("SELECT * FROM service_pricing WHERE is_active = 1");
    $pricing = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Document request error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Handle form submission
if (isset($_POST['submit_request'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        // Validate and sanitize input
        $serviceTypeId = (int)($_POST['service_type_id'] ?? 0);
        $officeId = !empty($_POST['office_id']) ? (int)$_POST['office_id'] : null;
        $documentType = Validator::validateText($_POST['document_type'] ?? '', 100);
        $documentDescription = Validator::validateText($_POST['document_description'] ?? '', 500);
        $purpose = Validator::validateText($_POST['purpose'] ?? '', 200);
        $destinationCountry = Validator::validateText($_POST['destination_country'] ?? '', 100);
        $urgencyLevel = Validator::validateText($_POST['urgency_level'] ?? '', 20);
        $notes = Validator::validateText($_POST['notes'] ?? '', 500);
        
        // Check for validation errors
        if (!$serviceTypeId) $errors[] = "Please select a service type";
        if (!$documentType) $errors[] = "Please specify the document type";
        if (!$purpose) $errors[] = "Please specify the purpose";
        if (!$destinationCountry) $errors[] = "Please specify the destination country";
        if (!in_array($urgencyLevel, ['normal', 'urgent', 'express'])) $errors[] = "Invalid urgency level";
        
        // Check if files were uploaded
        if (empty(array_filter($_FILES['documents']['name']))) {
            $errors[] = "Please upload at least one document";
        }
        
        // If no validation errors, proceed with request creation
        if (empty($errors)) {
            // Generate request number
            $requestNumber = 'DOC-' . date('Y') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
            
            // Calculate total amount based on pricing
            $totalAmount = 0;
            $stmt = $pdo->prepare("SELECT price FROM service_pricing WHERE service_type_id = ? AND office_id = ? AND urgency_level = ? AND is_active = 1");
            $stmt->execute([$serviceTypeId, $officeId, $urgencyLevel]);
            $pricingData = $stmt->fetch();
            
            if ($pricingData) {
                $totalAmount = $pricingData['price'];
            } else {
                // Fallback to base price
                $stmt = $pdo->prepare("SELECT base_price FROM document_service_types WHERE id = ?");
                $stmt->execute([$serviceTypeId]);
                $serviceData = $stmt->fetch();
                $totalAmount = $serviceData['base_price'] ?? 0;
            }
            
            // Create document request
            $stmt = $pdo->prepare("INSERT INTO document_requests (user_id, service_type_id, office_id, request_number, document_type, document_description, purpose, destination_country, urgency_level, status, total_amount, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', ?, NOW())");
            $stmt->execute([$currentUser['id'], $serviceTypeId, $officeId, $requestNumber, $documentType, $documentDescription, $purpose, $destinationCountry, $urgencyLevel, $totalAmount]);
            
            $requestId = $pdo->lastInsertId();
            
            // Process document uploads
            $uploadDir = '../uploads/document_services/' . $currentUser['firstname'] . '_' . $currentUser['lastname'] . '/';
            
            // Create directory if it doesn't exist
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $uploadSuccessCount = 0;
            
            foreach ($_FILES['documents']['tmp_name'] as $key => $value) {
                if ($_FILES['documents']['error'][$key] === UPLOAD_ERR_OK) {
                    $file = [
                        'name' => $_FILES['documents']['name'][$key],
                        'tmp_name' => $_FILES['documents']['tmp_name'][$key],
                        'size' => $_FILES['documents']['size'][$key],
                        'error' => $_FILES['documents']['error'][$key]
                    ];
                    
                    $validatedFile = Validator::validateFile($file);
                    
                    if ($validatedFile) {
                        $safeFilename = Validator::generateSafeFilename($validatedFile['name']);
                        $filepath = $uploadDir . $safeFilename;
                        
                        if (move_uploaded_file($validatedFile['tmp_name'], $filepath)) {
                            // Save document record
                            $stmt = $pdo->prepare("INSERT INTO officialisation_documents (request_id, user_id, document_name, document_type, original_filename, filename, file_path, file_size, mime_type, uploaded_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
                            $stmt->execute([
                                $requestId,
                                $currentUser['id'],
                                $documentType,
                                $documentType,
                                $validatedFile['name'],
                                $safeFilename,
                                $filepath,
                                $validatedFile['size'],
                                $validatedFile['type'] ?? 'application/octet-stream'
                            ]);
                            
                            $uploadSuccessCount++;
                        }
                    }
                }
            }
            
            // Add initial update
            $updateText = "Document request created with $uploadSuccessCount document(s) uploaded";
            $stmt = $pdo->prepare("INSERT INTO document_request_updates (request_id, user_id, update_text, update_type, created_at) VALUES (?, ?, ?, 'user', NOW())");
            $stmt->execute([$requestId, $currentUser['id'], $updateText]);
            
            $success = true;
            $successMessage = "Document request submitted successfully! Request number: $requestNumber";
            
            // Redirect to request details page
            header("Location: document-request-details.php?id=$requestId&success=1");
            exit();
        }
    } catch (Exception $e) {
        error_log("Document request submission error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Get pre-selected service if provided in URL
$selectedServiceId = isset($_GET['service_id']) ? (int)$_GET['service_id'] : 0;

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>New Document Request - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($currentUser['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($currentUser['firstname'] . ' ' . $currentUser['lastname']); ?></p>
                            <?php if (UserAuth::isAdmin()): ?>
                                <span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                    Admin
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">New Document Request</h1>
                    <p class="text-gray-600 mt-2">Submit a request for document legalization or officialisation</p>
                </div>
                <a href="document-services.php" class="text-gray-600 hover:text-gray-800">
                    <i class="fas fa-arrow-left mr-1"></i>
                    Back to Services
                </a>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if ($success): ?>
            <?php ErrorHandler::displaySuccess($successMessage); ?>
        <?php endif; ?>

        <!-- Request Form -->
        <div class="bg-white rounded-lg shadow">
            <div class="px-6 py-4 border-b border-gray-200">
                <h2 class="text-lg font-medium text-gray-900">Request Details</h2>
                <p class="text-sm text-gray-500 mt-1">Fill in the details below to submit your document request</p>
            </div>
            
            <form method="POST" action="" enctype="multipart/form-data" class="p-6">
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                
                <!-- Service Type Selection -->
                <div class="mb-6">
                    <?php if ($selectedServiceId): ?>
                        <?php 
                        // Get the selected service details
                        $selectedService = null;
                        foreach ($serviceTypes as $service) {
                            if ($service['id'] == $selectedServiceId) {
                                $selectedService = $service;
                                break;
                            }
                        }
                        ?>
                        
                        <?php if ($selectedService): ?>
                            <!-- Display selected service information -->
                            <label class="block text-sm font-medium text-gray-700 mb-2">Selected Service</label>
                            <div class="bg-blue-50 border border-blue-200 rounded-md p-4">
                                <div class="flex items-center justify-between">
                                    <div class="flex-1">
                                        <h3 class="text-lg font-medium text-gray-900"><?php echo htmlspecialchars($selectedService['name']); ?></h3>
                                        <p class="text-gray-600 mt-1"><?php echo htmlspecialchars($selectedService['description']); ?></p>
                                        <div class="flex items-center text-sm text-gray-500 mt-2">
                                            <i class="fas fa-clock mr-2"></i>
                                            <span><?php echo htmlspecialchars($selectedService['estimated_duration']); ?></span>
                                        </div>
                                    </div>
                                    <div class="text-right ml-6">
                                        <div class="text-2xl font-bold text-blue-600">₦<?php echo number_format($selectedService['base_price'], 2); ?></div>
                                        <div class="text-sm text-gray-500">Starting from</div>
                                    </div>
                                </div>
                            </div>
                            <input type="hidden" name="service_type_id" value="<?php echo $selectedServiceId; ?>">
                            
                            <!-- Change service link -->
                            <div class="mt-2 text-right">
                                <a href="document-services.php" class="text-sm text-blue-600 hover:text-blue-800">
                                    <i class="fas fa-edit mr-1"></i>
                                    Change service
                                </a>
                            </div>
                        <?php else: ?>
                            <!-- Fallback if service not found -->
                            <label for="service_type_id" class="block text-sm font-medium text-gray-700 mb-2">Service Type *</label>
                            <select name="service_type_id" id="service_type_id" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                                <option value="">Select a service type</option>
                                <?php foreach ($serviceTypes as $service): ?>
                                    <option value="<?php echo $service['id']; ?>" <?php echo $selectedServiceId === $service['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($service['name']); ?> - ₦<?php echo number_format($service['base_price'], 2); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        <?php endif; ?>
                    <?php else: ?>
                        <!-- No service pre-selected, show dropdown -->
                        <label for="service_type_id" class="block text-sm font-medium text-gray-700 mb-2">Service Type *</label>
                        <select name="service_type_id" id="service_type_id" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">Select a service type</option>
                            <?php foreach ($serviceTypes as $service): ?>
                                <option value="<?php echo $service['id']; ?>">
                                    <?php echo htmlspecialchars($service['name']); ?> - ₦<?php echo number_format($service['base_price'], 2); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    <?php endif; ?>
                </div>

                <!-- Office Selection (conditional) -->
                <div class="mb-6" id="officeSelection" style="display: none;">
                    <label for="office_id" class="block text-sm font-medium text-gray-700 mb-2">Processing Office</label>
                    <select name="office_id" id="office_id" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        <option value="">Select an office (optional)</option>
                        <?php foreach ($offices as $office): ?>
                            <option value="<?php echo $office['id']; ?>">
                                <?php echo htmlspecialchars($office['name']); ?> - <?php echo htmlspecialchars($office['city']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <!-- Document Information -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                    <div>
                        <label for="document_type" class="block text-sm font-medium text-gray-700 mb-2">Document Type *</label>
                        <input type="text" name="document_type" id="document_type" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" placeholder="e.g., Academic Transcript, Birth Certificate">
                    </div>

                    <div>
                        <label for="destination_country" class="block text-sm font-medium text-gray-700 mb-2">Destination Country *</label>
                        <input type="text" name="destination_country" id="destination_country" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" placeholder="e.g., United States, United Kingdom">
                    </div>
                </div>

                <div class="mb-6">
                    <label for="purpose" class="block text-sm font-medium text-gray-700 mb-2">Purpose *</label>
                    <input type="text" name="purpose" id="purpose" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" placeholder="e.g., University admission, Work permit, Visa application">
                </div>

                <div class="mb-6">
                    <label for="document_description" class="block text-sm font-medium text-gray-700 mb-2">Document Description</label>
                    <textarea name="document_description" id="document_description" rows="3" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" placeholder="Provide additional details about the documents..."></textarea>
                </div>

                <!-- Urgency Level -->
                <div class="mb-6">
                    <label for="urgency_level" class="block text-sm font-medium text-gray-700 mb-2">Urgency Level *</label>
                    <select name="urgency_level" id="urgency_level" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        <option value="normal">Normal (Standard processing time)</option>
                        <option value="urgent">Urgent (Faster processing, additional fee)</option>
                        <option value="express">Express (Fastest processing, premium fee)</option>
                    </select>
                </div>

                <!-- Document Upload -->
                <div class="mb-6">
                    <label for="documents" class="block text-sm font-medium text-gray-700 mb-2">Upload Documents *</label>
                    <div class="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-md">
                        <div class="space-y-1 text-center">
                            <i class="fas fa-cloud-upload-alt text-gray-400 text-3xl mb-4"></i>
                            <div class="flex text-sm text-gray-600">
                                <label for="documents" class="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-blue-500">
                                    <span>Upload files</span>
                                    <input id="documents" name="documents[]" type="file" class="sr-only" multiple accept=".pdf,.doc,.docx,.jpg,.jpeg,.png" onchange="updateFileList()" required>
                                </label>
                                <p class="pl-1">or drag and drop</p>
                            </div>
                            <p class="text-xs text-gray-500">PDF, DOC, DOCX, JPG, JPEG, PNG up to 10MB each</p>
                        </div>
                    </div>
                    <div id="file-list" class="mt-2 text-sm text-gray-600"></div>
                </div>

                <!-- Additional Notes -->
                <div class="mb-6">
                    <label for="notes" class="block text-sm font-medium text-gray-700 mb-2">Additional Notes</label>
                    <textarea name="notes" id="notes" rows="3" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" placeholder="Any special requirements or additional information..."></textarea>
                </div>

                <!-- Pricing Information -->
                <div class="mb-6 bg-gray-50 rounded-lg p-4">
                    <h3 class="text-sm font-medium text-gray-900 mb-2">Pricing Information</h3>
                    <div id="pricing-info" class="text-sm text-gray-600">
                        Select a service type to see pricing details
                    </div>
                </div>

                <!-- Form Actions -->
                <div class="flex justify-end space-x-4">
                    <a href="document-services.php" class="px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                        Cancel
                    </a>
                    <button type="submit" name="submit_request" class="px-6 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                        <i class="fas fa-paper-plane mr-2"></i>
                        Submit Request
                    </button>
                </div>
            </form>
        </div>

        <!-- Information Box -->
        <div class="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-info-circle text-blue-400"></i>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-blue-800">Important Information</h3>
                    <div class="mt-2 text-sm text-blue-700">
                        <ul class="list-disc list-inside space-y-1">
                            <li>Ensure all documents are clear and legible</li>
                            <li>Original documents may be required for certain services</li>
                            <li>Processing times may vary depending on the office and urgency level</li>
                            <li>You will be notified of any additional requirements or fees</li>
                            <li>Track your request status through the portal dashboard</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Pricing data from PHP
        const pricingData = <?php echo json_encode($pricing); ?>;
        const serviceTypes = <?php echo json_encode($serviceTypes); ?>;
        
        function updateFileList() {
            const fileInput = document.getElementById('documents');
            const fileList = document.getElementById('file-list');
            const files = fileInput.files;
            
            if (files.length > 0) {
                let fileListHTML = '<div class="mt-2"><strong>Selected files:</strong><ul class="list-disc list-inside mt-1">';
                for (let i = 0; i < files.length; i++) {
                    const file = files[i];
                    const size = (file.size / 1024).toFixed(1);
                    fileListHTML += `<li>${file.name} (${size} KB)</li>`;
                }
                fileListHTML += '</ul></div>';
                fileList.innerHTML = fileListHTML;
            } else {
                fileList.innerHTML = '';
            }
        }
        
        function updatePricing() {
            const serviceTypeId = document.getElementById('service_type_id').value;
            const officeId = document.getElementById('office_id').value;
            const urgencyLevel = document.getElementById('urgency_level').value;
            const pricingInfo = document.getElementById('pricing-info');
            
            if (!serviceTypeId) {
                pricingInfo.innerHTML = 'Select a service type to see pricing details';
                return;
            }
            
            // Find matching pricing
            let price = 0;
            let processingDays = 0;
            
            for (const pricing of pricingData) {
                if (pricing.service_type_id == serviceTypeId && 
                    (pricing.office_id == officeId || !pricing.office_id) && 
                    pricing.urgency_level === urgencyLevel) {
                    price = pricing.price;
                    processingDays = pricing.processing_days;
                    break;
                }
            }
            
            // Fallback to base price
            if (price === 0) {
                const service = serviceTypes.find(s => s.id == serviceTypeId);
                if (service) {
                    price = service.base_price;
                }
            }
            
            pricingInfo.innerHTML = `
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <strong>Estimated Cost:</strong> ₦${price.toLocaleString()}
                    </div>
                    <div>
                        <strong>Processing Time:</strong> ${processingDays > 0 ? processingDays + ' business days' : 'Varies by service'}
                    </div>
                </div>
            `;
        }
        
        function toggleOfficeSelection() {
            const serviceTypeId = document.getElementById('service_type_id').value;
            const officeSelection = document.getElementById('officeSelection');
            
            // Show office selection for embassy-related services
            if (serviceTypeId == 2) { // Embassy Legalization
                officeSelection.style.display = 'block';
            } else {
                officeSelection.style.display = 'none';
                document.getElementById('office_id').value = '';
            }
        }
        
        // Event listeners
        document.getElementById('service_type_id').addEventListener('change', function() {
            toggleOfficeSelection();
            updatePricing();
        });
        
        document.getElementById('office_id').addEventListener('change', updatePricing);
        document.getElementById('urgency_level').addEventListener('change', updatePricing);
        
        // Initialize pricing on page load
        if (document.getElementById('service_type_id').value) {
            toggleOfficeSelection();
            updatePricing();
        }
        
        // If service is pre-selected from URL, initialize pricing
        <?php if ($selectedServiceId && $selectedService): ?>
        document.addEventListener('DOMContentLoaded', function() {
            updatePricing();
        });
        <?php endif; ?>
        
        // Drag and drop functionality
        const dropZone = document.querySelector('input[type="file"]').parentElement.parentElement.parentElement;
        
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, preventDefaults, false);
        });

        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }

        ['dragenter', 'dragover'].forEach(eventName => {
            dropZone.addEventListener(eventName, highlight, false);
        });

        ['dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, unhighlight, false);
        });

        function highlight(e) {
            dropZone.classList.add('border-blue-400', 'bg-blue-50');
        }

        function unhighlight(e) {
            dropZone.classList.remove('border-blue-400', 'bg-blue-50');
        }

        dropZone.addEventListener('drop', handleDrop, false);

        function handleDrop(e) {
            const dt = e.dataTransfer;
            const files = dt.files;
            document.getElementById('documents').files = files;
            updateFileList();
        }
    </script>
</body>
</html> 