-- Document Officialisation Service Tables
-- This file contains all the necessary tables for document legalization and officialisation services

-- Document service types table
CREATE TABLE IF NOT EXISTS document_service_types (
    id INT NOT NULL AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    estimated_duration VARCHAR(50),
    base_price DECIMAL(10,2) DEFAULT 0.00,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id)
);

-- Embassies and ministries table
CREATE TABLE IF NOT EXISTS official_offices (
    id INT NOT NULL AUTO_INCREMENT,
    name VARCHAR(200) NOT NULL,
    type ENUM('embassy', 'ministry', 'consulate', 'other') NOT NULL,
    country VARCHAR(100) NOT NULL,
    city VARCHAR(100),
    address TEXT,
    phone VARCHAR(50),
    email VARCHAR(100),
    website VARCHAR(200),
    processing_time VARCHAR(50),
    requirements TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id)
);

-- Document officialisation requests table
CREATE TABLE IF NOT EXISTS document_requests (
    id INT NOT NULL AUTO_INCREMENT,
    user_id INT NOT NULL,
    service_type_id INT NOT NULL,
    office_id INT,
    request_number VARCHAR(50) UNIQUE,
    document_type VARCHAR(100) NOT NULL,
    document_description TEXT,
    purpose VARCHAR(200),
    destination_country VARCHAR(100),
    urgency_level ENUM('normal', 'urgent', 'express') DEFAULT 'normal',
    status ENUM('pending', 'submitted', 'in_progress', 'completed', 'rejected', 'cancelled') DEFAULT 'pending',
    total_amount DECIMAL(10,2) DEFAULT 0.00,
    paid_amount DECIMAL(10,2) DEFAULT 0.00,
    payment_status ENUM('pending', 'partial', 'paid') DEFAULT 'pending',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL,
    PRIMARY KEY (id),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (service_type_id) REFERENCES document_service_types(id) ON DELETE RESTRICT,
    FOREIGN KEY (office_id) REFERENCES official_offices(id) ON DELETE SET NULL
);

-- Documents for officialisation table
CREATE TABLE IF NOT EXISTS officialisation_documents (
    id INT NOT NULL AUTO_INCREMENT,
    request_id INT NOT NULL,
    user_id INT NOT NULL,
    document_name VARCHAR(200) NOT NULL,
    document_type VARCHAR(100) NOT NULL,
    original_filename VARCHAR(255) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_size INT NOT NULL,
    mime_type VARCHAR(100),
    is_original BOOLEAN DEFAULT FALSE,
    is_certified_copy BOOLEAN DEFAULT FALSE,
    uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    verified_at TIMESTAMP NULL,
    verified_by INT NULL,
    PRIMARY KEY (id),
    FOREIGN KEY (request_id) REFERENCES document_requests(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (verified_by) REFERENCES users(id) ON DELETE SET NULL
);

-- Document request updates/timeline table
CREATE TABLE IF NOT EXISTS document_request_updates (
    id INT NOT NULL AUTO_INCREMENT,
    request_id INT NOT NULL,
    user_id INT NOT NULL,
    update_text TEXT NOT NULL,
    update_type ENUM('user', 'admin', 'system', 'office') DEFAULT 'user',
    status_change VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    FOREIGN KEY (request_id) REFERENCES document_requests(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Document request payments table
CREATE TABLE IF NOT EXISTS document_request_payments (
    id INT NOT NULL AUTO_INCREMENT,
    request_id INT NOT NULL,
    user_id INT NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    payment_method ENUM('bank_transfer', 'card', 'cash', 'other') DEFAULT 'bank_transfer',
    payment_reference VARCHAR(100),
    payment_date TIMESTAMP NULL,
    status ENUM('pending', 'completed', 'failed', 'refunded') DEFAULT 'pending',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    FOREIGN KEY (request_id) REFERENCES document_requests(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Service pricing table
CREATE TABLE IF NOT EXISTS service_pricing (
    id INT NOT NULL AUTO_INCREMENT,
    service_type_id INT NOT NULL,
    office_id INT,
    urgency_level ENUM('normal', 'urgent', 'express') DEFAULT 'normal',
    price DECIMAL(10,2) NOT NULL,
    processing_days INT NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    FOREIGN KEY (service_type_id) REFERENCES document_service_types(id) ON DELETE CASCADE,
    FOREIGN KEY (office_id) REFERENCES official_offices(id) ON DELETE CASCADE
);

-- Insert default service types
INSERT INTO document_service_types (name, description, estimated_duration, base_price) VALUES
('Ministry of Foreign Affairs Legalization', 'Legalization of documents at the Ministry of Foreign Affairs for international use', '5-7 business days', 150.00),
('Embassy Legalization', 'Legalization of documents at foreign embassies and consulates', '7-14 business days', 200.00),
('Apostille Service', 'Apostille certification for Hague Convention countries', '3-5 business days', 120.00),
('Document Translation', 'Official translation of documents with certification', '2-3 business days', 80.00),
('Document Certification', 'Certification of document copies as true copies', '1-2 business days', 50.00),
('Notary Services', 'Notarization of documents and affidavits', '1 business day', 100.00),
('Consular Services', 'Various consular services and document processing', '5-10 business days', 180.00);

-- Insert sample official offices
INSERT INTO official_offices (name, type, country, city, address, processing_time, requirements) VALUES
('Ministry of Foreign Affairs', 'ministry', 'Nigeria', 'Abuja', 'Federal Secretariat Complex, Shehu Shagari Way, Central Business District', '5-7 business days', 'Original documents, application form, payment receipt'),
('US Embassy', 'embassy', 'United States', 'Lagos', '2 Walter Carrington Crescent, Victoria Island', '7-14 business days', 'Appointment required, original documents, valid passport'),
('UK High Commission', 'embassy', 'United Kingdom', 'Lagos', '19 Torrens Close, Ikoyi', '7-10 business days', 'Online application, original documents, payment confirmation'),
('Canadian High Commission', 'embassy', 'Canada', 'Lagos', '4 Anifowoshe Street, Victoria Island', '10-15 business days', 'Appointment booking, original documents, processing fee'),
('German Embassy', 'embassy', 'Germany', 'Abuja', '9 Lake Maracaibo Close, Maitama', '5-8 business days', 'Original documents, application form, payment receipt');

-- Insert sample pricing
INSERT INTO service_pricing (service_type_id, office_id, urgency_level, price, processing_days) VALUES
(1, 1, 'normal', 150.00, 7),
(1, 1, 'urgent', 250.00, 3),
(1, 1, 'express', 350.00, 1),
(2, 2, 'normal', 200.00, 14),
(2, 2, 'urgent', 300.00, 7),
(2, 3, 'normal', 180.00, 10),
(2, 3, 'urgent', 280.00, 5),
(3, 1, 'normal', 120.00, 5),
(3, 1, 'urgent', 200.00, 2),
(4, NULL, 'normal', 80.00, 3),
(4, NULL, 'urgent', 120.00, 1),
(5, NULL, 'normal', 50.00, 2),
(5, NULL, 'urgent', 80.00, 1);

-- Create indexes for better performance
CREATE INDEX idx_document_requests_user_id ON document_requests(user_id);
CREATE INDEX idx_document_requests_status ON document_requests(status);
CREATE INDEX idx_document_requests_created_at ON document_requests(created_at);
CREATE INDEX idx_officialisation_documents_request_id ON officialisation_documents(request_id);
CREATE INDEX idx_document_request_updates_request_id ON document_request_updates(request_id);
CREATE INDEX idx_document_request_payments_request_id ON document_request_payments(request_id);
CREATE INDEX idx_service_pricing_service_type_id ON service_pricing(service_type_id); 