<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get request ID
$requestId = $_GET['request_id'] ?? 0;

if (!$requestId) {
    header('Location: document-services.php');
    exit();
}

// Get user data and request details
try {
    $pdo = getDatabaseConnection();
    
    // Get user data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
    
    // Get document request details
    $stmt = $pdo->prepare("
        SELECT dr.*, dst.name as service_name, oo.name as office_name
        FROM document_requests dr
        LEFT JOIN document_service_types dst ON dr.service_type_id = dst.id
        LEFT JOIN official_offices oo ON dr.office_id = oo.id
        WHERE dr.id = ? AND dr.user_id = ?
    ");
    $stmt->execute([$requestId, $_SESSION['user_id']]);
    $request = $stmt->fetch();
    
    if (!$request) {
        header('Location: document-services.php');
        exit();
    }
    
    // Check if already submitted
    if ($request['status'] !== 'pending') {
        header('Location: document-request-details.php?id=' . $requestId);
        exit();
    }
    
} catch (Exception $e) {
    error_log("Submit order error: " . $e->getMessage());
    $error = "An error occurred. Please try again later.";
}

// Handle order submission
if (isset($_POST['submit_order'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        // Update request status to submitted
        $stmt = $pdo->prepare("UPDATE document_requests SET status = 'submitted', updated_at = NOW() WHERE id = ? AND user_id = ?");
        $stmt->execute([$requestId, $_SESSION['user_id']]);
        
        // Add update to timeline
        $updateText = "Order submitted for review and invoicing";
        $stmt = $pdo->prepare("INSERT INTO document_request_updates (request_id, user_id, update_text, update_type, status_change, created_at) VALUES (?, ?, ?, 'user', 'submitted', NOW())");
        $stmt->execute([$requestId, $_SESSION['user_id'], $updateText]);
        
        // Send notification email to admin (optional)
        try {
            $emailService = new EmailService();
            $emailService->sendOrderSubmittedNotification($request, $user);
        } catch (Exception $e) {
            error_log("Failed to send order notification email: " . $e->getMessage());
        }
        
        // Redirect to document request details with success message
        $_SESSION['success'] = "Order submitted successfully! We'll review your request and send you an invoice shortly.";
        header('Location: document-request-details.php?id=' . $requestId . '&success=1');
        exit();
        
    } catch (Exception $e) {
        error_log("Order submission error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Submit Order - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($user['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Submit Order</h1>
                    <p class="text-gray-600 mt-2">Review and submit your document request for processing</p>
                </div>
                <a href="document-request-details.php?id=<?php echo $requestId; ?>" class="text-gray-600 hover:text-gray-800">
                    <i class="fas fa-arrow-left mr-1"></i>
                    Back to Request
                </a>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>



        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <!-- Order Summary -->
            <div class="lg:col-span-2">
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-medium text-gray-900">Order Summary</h2>
                    </div>
                    <div class="px-6 py-4">
                        <dl class="grid grid-cols-1 gap-x-4 gap-y-6 sm:grid-cols-2">
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Request Number</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['request_number']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Service Type</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['service_name']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Document Type</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['document_type']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Destination Country</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['destination_country']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Urgency Level</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo ucfirst($request['urgency_level']); ?></dd>
                            </div>
                            <div>
                                <dt class="text-sm font-medium text-gray-500">Estimated Amount</dt>
                                <dd class="mt-1 text-sm text-gray-900">₦<?php echo number_format($request['total_amount'], 2); ?></dd>
                            </div>
                            <div class="sm:col-span-2">
                                <dt class="text-sm font-medium text-gray-500">Document Description</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo nl2br(htmlspecialchars($request['document_description'])); ?></dd>
                            </div>
                            <?php if ($request['purpose']): ?>
                            <div class="sm:col-span-2">
                                <dt class="text-sm font-medium text-gray-500">Purpose</dt>
                                <dd class="mt-1 text-sm text-gray-900"><?php echo htmlspecialchars($request['purpose']); ?></dd>
                            </div>
                            <?php endif; ?>
                        </dl>
                    </div>
                </div>

                <!-- Important Information -->
                <div class="mt-6 bg-blue-50 border border-blue-200 rounded-lg p-6">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <i class="fas fa-info-circle text-blue-400"></i>
                        </div>
                        <div class="ml-3">
                            <h3 class="text-sm font-medium text-blue-800">Important Information</h3>
                            <div class="mt-2 text-sm text-blue-700">
                                <ul class="list-disc list-inside space-y-1">
                                    <li>Your order will be reviewed by our team to determine the exact cost</li>
                                    <li>We'll send you a detailed invoice with the final pricing</li>
                                    <li>Processing will begin once payment is confirmed</li>
                                    <li>You can track the progress in your request details</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Submit Order -->
            <div class="space-y-6">
                <!-- Order Actions -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Submit Order</h3>
                    </div>
                    <div class="px-6 py-4">
                        <form method="POST">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                            
                            <div class="mb-6">
                                <p class="text-sm text-gray-600 mb-4">
                                    By submitting this order, you confirm that all information provided is accurate and complete.
                                </p>
                                
                                <div class="flex items-start">
                                    <div class="flex items-center h-5">
                                        <input id="confirm" name="confirm" type="checkbox" required class="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 rounded">
                                    </div>
                                    <div class="ml-3 text-sm">
                                        <label for="confirm" class="font-medium text-gray-700">I confirm the information is correct</label>
                                        <p class="text-gray-500">Please review all details before submitting</p>
                                    </div>
                                </div>
                            </div>
                            
                            <button type="submit" name="submit_order" class="w-full inline-flex justify-center items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                <i class="fas fa-paper-plane mr-2"></i>
                                Submit Order for Review
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Contact Information -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Need Help?</h3>
                    </div>
                    <div class="px-6 py-4">
                        <div class="space-y-3">
                            <div class="flex items-center">
                                <i class="fas fa-envelope text-gray-400 mr-3"></i>
                                <div>
                                    <p class="text-sm font-medium text-gray-900">Email Support</p>
                                    <p class="text-sm text-gray-500">services@migration.ng</p>
                                </div>
                            </div>
                            <div class="flex items-center">
                                <i class="fas fa-phone text-gray-400 mr-3"></i>
                                <div>
                                    <p class="text-sm font-medium text-gray-900">Phone Support</p>
                                    <p class="text-sm text-gray-500">+234 XXX XXX XXXX</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Auto-hide success messages after 5 seconds
        setTimeout(function() {
            const successMessages = document.querySelectorAll('.bg-green-50');
            successMessages.forEach(function(message) {
                message.style.display = 'none';
            });
        }, 5000);
    </script>
</body>
</html> 