<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || !UserAuth::isAdmin()) {
    header('Location: login.php');
    exit();
}

// Check if this is a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: admin-applications.php');
    exit();
}

// Validate required fields
$application_id = $_POST['application_id'] ?? '';
$status = $_POST['status'] ?? '';
$note = $_POST['note'] ?? '';

if (empty($application_id) || empty($status)) {
    $_SESSION['error'] = "Application ID and status are required.";
    header('Location: admin-applications.php');
    exit();
}

// Validate status
$valid_statuses = ['pending', 'in_progress', 'completed', 'rejected'];
if (!in_array($status, $valid_statuses)) {
    $_SESSION['error'] = "Invalid status provided.";
    header('Location: admin-applications.php');
    exit();
}

try {
    $pdo = getDatabaseConnection();
    
    // Get application details with user information
    $stmt = $pdo->prepare("
        SELECT 
            pa.*,
            u.firstname,
            u.lastname,
            u.email
        FROM portal_applications pa
        LEFT JOIN users u ON pa.user_id = u.id
        WHERE pa.id = ?
    ");
    $stmt->execute([$application_id]);
    $application = $stmt->fetch();
    
    if (!$application) {
        $_SESSION['error'] = "Application not found.";
        header('Location: admin-applications.php');
        exit();
    }
    
    // Get admin information
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $admin = $stmt->fetch();
    
    // Update application status
    $stmt = $pdo->prepare("
        UPDATE portal_applications 
        SET status = ?, updated_at = NOW() 
        WHERE id = ?
    ");
    $stmt->execute([$status, $application_id]);
    
    // Add status update to timeline/notes if note is provided
    if (!empty($note)) {
        // You might want to create a separate table for application notes/timeline
        // For now, we'll just log it
        error_log("Application {$application_id} status updated to {$status} by admin {$_SESSION['user_id']}. Note: {$note}");
    }
    
    // Send email notification to user
    $statusText = [
        'pending' => 'Pending',
        'in_progress' => 'In Progress', 
        'completed' => 'Completed',
        'rejected' => 'Rejected'
    ];
    
    $subject = "Application Status Update - Migration.ng";
    $message = EmailService::buildApplicationStatusUpdateEmail($application, $statusText[$status], $note, $admin);
    
    $emailSent = EmailService::sendEmail($application['email'], $subject, $message);
    
    if ($emailSent) {
        $_SESSION['success'] = "Application status updated successfully and notification sent to user.";
    } else {
        $_SESSION['success'] = "Application status updated successfully, but email notification failed.";
    }
    
} catch (Exception $e) {
    error_log("Update application status error: " . $e->getMessage());
    $_SESSION['error'] = "An error occurred while updating the application status.";
}

// Redirect back to admin applications page
header('Location: admin-applications.php');
exit();
?> 