<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || !UserAuth::isAdmin()) {
    header('Location: login.php');
    exit();
}

// Check if this is a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: admin-applications.php');
    exit();
}

// Validate required fields
$document_id = $_POST['document_id'] ?? '';
$status = $_POST['status'] ?? '';
$final_price = $_POST['final_price'] ?? '';
$note = $_POST['note'] ?? '';

if (empty($document_id) || empty($status)) {
    $_SESSION['error'] = "Document ID and status are required.";
    header('Location: admin-applications.php');
    exit();
}

// Validate status
$valid_statuses = ['pending', 'submitted', 'in_progress', 'completed', 'cancelled'];
if (!in_array($status, $valid_statuses)) {
    $_SESSION['error'] = "Invalid status provided.";
    header('Location: admin-applications.php');
    exit();
}

try {
    $pdo = getDatabaseConnection();
    
    // Get document request details with user information
    $stmt = $pdo->prepare("
        SELECT 
            dr.*,
            u.firstname,
            u.lastname,
            u.email,
            dst.name as service_name
        FROM document_requests dr
        LEFT JOIN users u ON dr.user_id = u.id
        LEFT JOIN document_service_types dst ON dr.service_type_id = dst.id
        WHERE dr.id = ?
    ");
    $stmt->execute([$document_id]);
    $documentRequest = $stmt->fetch();
    
    if (!$documentRequest) {
        $_SESSION['error'] = "Document request not found.";
        header('Location: admin-applications.php');
        exit();
    }
    
    // Get admin information
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $admin = $stmt->fetch();
    
    // Update document request
    $updateFields = ["status = ?", "updated_at = NOW()"];
    $params = [$status, $document_id];
    
    // If final price is provided, update it
    if (!empty($final_price) && is_numeric($final_price)) {
        $updateFields[] = "total_amount = ?";
        $params = [$status, $final_price, $document_id];
    }
    
    $stmt = $pdo->prepare("
        UPDATE document_requests 
        SET " . implode(", ", $updateFields) . "
        WHERE id = ?
    ");
    $stmt->execute($params);
    
    // Add update to timeline
    $updateText = "Status updated to " . ucfirst(str_replace('_', ' ', $status));
    if (!empty($final_price) && is_numeric($final_price)) {
        $updateText .= " and final price set to ₦" . number_format($final_price, 2);
    }
    if (!empty($note)) {
        $updateText .= ". Note: " . $note;
    }
    
    $stmt = $pdo->prepare("
        INSERT INTO document_request_updates 
        (request_id, user_id, update_text, update_type, status_change, created_at) 
        VALUES (?, ?, ?, 'admin', ?, NOW())
    ");
    $stmt->execute([$document_id, $_SESSION['user_id'], $updateText, $status]);
    

    
    // Send email notification to user about the update
    try {
        EmailService::sendDocumentRequestUpdateNotification($documentRequest, $admin);
    } catch (Exception $e) {
        error_log("Failed to send update notification email: " . $e->getMessage());
    }
    
    $_SESSION['success'] = "Document request updated successfully and notification sent to user.";
    
} catch (Exception $e) {
    error_log("Update document request error: " . $e->getMessage());
    $_SESSION['error'] = "An error occurred while updating the document request.";
}

// Redirect back to admin applications page
header('Location: admin-applications.php');
exit();
?> 