<?php
// Start session for user authentication
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/UserAuth.php';

// Initialize error handling
ErrorHandler::init();

// Check if user is logged in
UserAuth::requireLogin();

// Get user data
$currentUser = UserAuth::getCurrentUser();

// Get application ID from URL
$applicationId = isset($_GET['application_id']) ? (int)$_GET['application_id'] : 0;

if (!$applicationId) {
    header('Location: applications.php?error=invalid_id');
    exit();
}

// Get application data
try {
    $pdo = getDatabaseConnection();
    $stmt = $pdo->prepare("SELECT * FROM portal_applications WHERE id = ? AND user_id = ?");
    $stmt->execute([$applicationId, $currentUser['id']]);
    $application = $stmt->fetch();
    
    if (!$application) {
        header('Location: applications.php?error=not_found');
        exit();
    }
    
    // Get existing documents for this application
    $stmt = $pdo->prepare("SELECT * FROM application_documents WHERE application_id = ? AND user_id = ? ORDER BY uploaded_at DESC");
    $stmt->execute([$applicationId, $currentUser['id']]);
    $existingDocuments = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Upload documents error: " . $e->getMessage());
    header('Location: applications.php?error=database_error');
    exit();
}

// Initialize variables
$errors = [];
$success = false;

// Handle document upload
if (isset($_POST['upload_documents'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        $documentType = Validator::validateText($_POST['document_type'] ?? '', 50);
        
        if (!$documentType) {
            $errors[] = "Please select a document type";
        }
        
        // Check if files were uploaded
        if (empty(array_filter($_FILES['documents']['name']))) {
            $errors[] = "Please select at least one file to upload";
        }
        
        // If no validation errors, proceed with file upload
        if (empty($errors)) {
            $uploadDir = '../uploads/' . $currentUser['firstname'] . '_' . $currentUser['lastname'] . '/';
            
            // Create directory if it doesn't exist
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $uploadSuccessCount = 0;
            $uploadErrors = [];
            
            // Process file uploads
            foreach ($_FILES['documents']['tmp_name'] as $key => $value) {
                if ($_FILES['documents']['error'][$key] === UPLOAD_ERR_OK) {
                    $file = [
                        'name' => $_FILES['documents']['name'][$key],
                        'tmp_name' => $_FILES['documents']['tmp_name'][$key],
                        'size' => $_FILES['documents']['size'][$key],
                        'error' => $_FILES['documents']['error'][$key]
                    ];
                    
                    $validatedFile = Validator::validateFile($file);
                    
                    if ($validatedFile) {
                        $safeFilename = Validator::generateSafeFilename($validatedFile['name']);
                        $filepath = $uploadDir . $safeFilename;
                        
                        if (move_uploaded_file($validatedFile['tmp_name'], $filepath)) {
                            // Save document record to database
                            $stmt = $pdo->prepare("INSERT INTO application_documents (user_id, application_id, document_type, filename, original_filename, file_size, file_path, uploaded_at) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())");
                            $stmt->execute([
                                $currentUser['id'],
                                $applicationId,
                                $documentType,
                                $safeFilename,
                                $validatedFile['name'],
                                $validatedFile['size'],
                                $filepath
                            ]);
                            
                            $uploadSuccessCount++;
                        } else {
                            $uploadErrors[] = "Failed to upload " . $validatedFile['name'];
                        }
                    } else {
                        $uploadErrors[] = "Invalid file: " . $file['name'];
                    }
                } else {
                    $uploadErrors[] = "Upload error for file: " . $_FILES['documents']['name'][$key];
                }
            }
            
            if ($uploadSuccessCount > 0) {
                $success = true;
                $successMessage = "Successfully uploaded $uploadSuccessCount document(s).";
                if (!empty($uploadErrors)) {
                    $successMessage .= " Some files failed to upload: " . implode(', ', $uploadErrors);
                }
                
                // Add update to application_updates table
                $updateText = "User uploaded $uploadSuccessCount document(s)";
                $stmt = $pdo->prepare("INSERT INTO application_updates (application_id, user_id, update_text, update_type, created_at) VALUES (?, ?, ?, 'user', NOW())");
                $stmt->execute([$applicationId, $currentUser['id'], $updateText]);
                
                // Refresh documents list
                $stmt = $pdo->prepare("SELECT * FROM application_documents WHERE application_id = ? AND user_id = ? ORDER BY uploaded_at DESC");
                $stmt->execute([$applicationId, $currentUser['id']]);
                $existingDocuments = $stmt->fetchAll();
            } else {
                $errors[] = "No documents were uploaded successfully.";
            }
        }
    } catch (Exception $e) {
        error_log("Document upload error: " . $e->getMessage());
        $errors[] = "An error occurred. Please try again later.";
    }
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();

// Document types
$documentTypes = [
    'transcript' => 'Academic Transcript',
    'certificate' => 'Certificate/Diploma',
    'passport' => 'Passport',
    'cv' => 'CV/Resume',
    'personal_statement' => 'Personal Statement',
    'reference_letter' => 'Reference Letter',
    'english_test' => 'English Test Result',
    'financial_statement' => 'Financial Statement',
    'other' => 'Other Document'
];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload Documents - Migration.ng Portal</title>
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    <!-- Navigation -->
    <nav class="bg-white shadow-sm border-b">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="flex items-center">
                        <img src="/assets/images/m..png" alt="Migration.ng" class="h-8 w-auto">
                        <span class="ml-2 text-xl font-semibold text-gray-900">Portal</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-home"></i>
                    </a>
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                            <span class="text-white text-sm font-medium"><?php echo strtoupper(substr($currentUser['firstname'], 0, 1)); ?></span>
                        </div>
                        <div class="hidden md:block">
                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($currentUser['firstname'] . ' ' . $currentUser['lastname']); ?></p>
                            <?php if (UserAuth::isAdmin()): ?>
                                <span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                    Admin
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900">Upload Documents</h1>
                    <p class="text-gray-600 mt-2">Upload documents for your application</p>
                    <div class="mt-2 text-sm text-gray-500">
                        Application: <?php echo htmlspecialchars($application['university']); ?> - <?php echo htmlspecialchars(html_entity_decode($application['degree'])); ?> in <?php echo htmlspecialchars($application['program']); ?>
                    </div>
                </div>
                <div class="flex space-x-3">
                    <a href="application-details.php?id=<?php echo $application['id']; ?>" class="text-blue-600 hover:text-blue-800">
                        <i class="fas fa-eye mr-1"></i>
                        View Application
                    </a>
                    <a href="applications.php" class="text-gray-600 hover:text-gray-800">
                        <i class="fas fa-arrow-left mr-1"></i>
                        Back to Applications
                    </a>
                </div>
            </div>
        </div>

        <!-- Display error messages -->
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <?php ErrorHandler::displayValidationError($error); ?>
            <?php endforeach; ?>
        <?php endif; ?>

        <!-- Display success message -->
        <?php if ($success): ?>
            <?php ErrorHandler::displaySuccess($successMessage); ?>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <!-- Upload Form -->
            <div class="bg-white rounded-lg shadow">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h2 class="text-lg font-medium text-gray-900">Upload New Documents</h2>
                    <p class="text-sm text-gray-500 mt-1">Select document type and files to upload</p>
                </div>
                
                <form method="POST" action="" enctype="multipart/form-data" class="p-6">
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                    
                    <!-- Document Type -->
                    <div class="mb-6">
                        <label for="document_type" class="block text-sm font-medium text-gray-700 mb-2">Document Type *</label>
                        <select name="document_type" id="document_type" required class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                            <option value="">Select document type</option>
                            <?php foreach ($documentTypes as $value => $label): ?>
                                <option value="<?php echo htmlspecialchars($value); ?>"><?php echo htmlspecialchars($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- File Upload -->
                    <div class="mb-6">
                        <label for="documents" class="block text-sm font-medium text-gray-700 mb-2">Select Files *</label>
                        <div class="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-md">
                            <div class="space-y-1 text-center">
                                <i class="fas fa-cloud-upload-alt text-gray-400 text-3xl mb-4"></i>
                                <div class="flex text-sm text-gray-600">
                                    <label for="documents" class="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-blue-500">
                                        <span>Upload files</span>
                                        <input id="documents" name="documents[]" type="file" class="sr-only" multiple accept=".pdf,.doc,.docx,.jpg,.jpeg,.png" onchange="updateFileList()">
                                    </label>
                                    <p class="pl-1">or drag and drop</p>
                                </div>
                                <p class="text-xs text-gray-500">PDF, DOC, DOCX, JPG, JPEG, PNG up to 10MB each</p>
                            </div>
                        </div>
                        <div id="file-list" class="mt-2 text-sm text-gray-600"></div>
                    </div>

                    <!-- Upload Button -->
                    <div class="flex justify-end">
                        <button
                            type="submit"
                            name="upload_documents"
                            class="px-6 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                        >
                            <i class="fas fa-upload mr-2"></i>
                            Upload Documents
                        </button>
                    </div>
                </form>
            </div>

            <!-- Existing Documents -->
            <div class="bg-white rounded-lg shadow">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h2 class="text-lg font-medium text-gray-900">Existing Documents</h2>
                    <p class="text-sm text-gray-500 mt-1">Documents already uploaded for this application</p>
                </div>
                
                <div class="p-6">
                    <?php if (empty($existingDocuments)): ?>
                        <div class="text-center py-8">
                            <i class="fas fa-file-alt text-gray-400 text-4xl mb-4"></i>
                            <p class="text-gray-500">No documents uploaded yet</p>
                            <p class="text-sm text-gray-400 mt-1">Upload your first document using the form on the left</p>
                        </div>
                    <?php else: ?>
                        <div class="space-y-4">
                            <?php foreach ($existingDocuments as $doc): ?>
                                <div class="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                                    <div class="flex items-center space-x-3">
                                        <div class="flex-shrink-0">
                                            <i class="fas fa-file-alt text-blue-500 text-xl"></i>
                                        </div>
                                        <div>
                                            <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($doc['original_filename']); ?></p>
                                            <p class="text-xs text-gray-500">
                                                <?php echo htmlspecialchars($documentTypes[$doc['document_type']] ?? $doc['document_type']); ?> • 
                                                <?php echo number_format($doc['file_size'] / 1024, 1); ?> KB • 
                                                <?php echo date('M j, Y', strtotime($doc['uploaded_at'])); ?>
                                            </p>
                                        </div>
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <?php if ($doc['is_verified']): ?>
                                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                                <i class="fas fa-check mr-1"></i>Verified
                                            </span>
                                        <?php else: ?>
                                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                                                <i class="fas fa-clock mr-1"></i>Pending
                                            </span>
                                        <?php endif; ?>
                                        <a href="download-document.php?id=<?php echo $doc['id']; ?>" class="text-blue-600 hover:text-blue-800" title="Download">
                                            <i class="fas fa-download"></i>
                                        </a>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Information Box -->
        <div class="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-info-circle text-blue-400"></i>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-blue-800">Document Upload Guidelines</h3>
                    <div class="mt-2 text-sm text-blue-700">
                        <ul class="list-disc list-inside space-y-1">
                            <li>Accepted file formats: PDF, DOC, DOCX, JPG, JPEG, PNG</li>
                            <li>Maximum file size: 10MB per file</li>
                            <li>You can upload multiple files at once</li>
                            <li>Documents will be reviewed by our team within 24-48 hours</li>
                            <li>Keep original file names for easier identification</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        function updateFileList() {
            const fileInput = document.getElementById('documents');
            const fileList = document.getElementById('file-list');
            const files = fileInput.files;
            
            if (files.length > 0) {
                let fileListHTML = '<div class="mt-2"><strong>Selected files:</strong><ul class="list-disc list-inside mt-1">';
                for (let i = 0; i < files.length; i++) {
                    const file = files[i];
                    const size = (file.size / 1024).toFixed(1);
                    fileListHTML += `<li>${file.name} (${size} KB)</li>`;
                }
                fileListHTML += '</ul></div>';
                fileList.innerHTML = fileListHTML;
            } else {
                fileList.innerHTML = '';
            }
        }

        // Drag and drop functionality
        const dropZone = document.querySelector('input[type="file"]').parentElement.parentElement.parentElement;
        
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, preventDefaults, false);
        });

        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }

        ['dragenter', 'dragover'].forEach(eventName => {
            dropZone.addEventListener(eventName, highlight, false);
        });

        ['dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, unhighlight, false);
        });

        function highlight(e) {
            dropZone.classList.add('border-blue-400', 'bg-blue-50');
        }

        function unhighlight(e) {
            dropZone.classList.remove('border-blue-400', 'bg-blue-50');
        }

        dropZone.addEventListener('drop', handleDrop, false);

        function handleDrop(e) {
            const dt = e.dataTransfer;
            const files = dt.files;
            document.getElementById('documents').files = files;
            updateFileList();
        }
    </script>
</body>
</html> 