<?php
// Start session
session_start();

// Define environment
define('ENVIRONMENT', 'production');

// Include required files
require_once '../config/database.php';
require_once '../includes/Validator.php';
require_once '../includes/ErrorHandler.php';
require_once '../includes/EmailService.php';
require_once '../includes/EmailVerification.php';

// Initialize error handling
ErrorHandler::init();

// Initialize variables
$verificationStatus = '';
$verificationMessage = '';
$showResendForm = false;
$success = false;

// Handle email verification
if (isset($_GET['token'])) {
    $token = trim($_GET['token']);
    
    if (!empty($token)) {
        // Verify the token
        $user = EmailVerification::verifyToken($token);
        
        if ($user) {
            $success = true;
            $verificationStatus = 'success';
            $verificationMessage = "Email verification successful! Your account has been activated. You can now log in to your account.";
        } else {
            $verificationStatus = 'error';
            $verificationMessage = EmailVerification::getVerificationStatusMessage($token);
            $showResendForm = true;
        }
    } else {
        $verificationStatus = 'error';
        $verificationMessage = "Invalid verification link.";
        $showResendForm = true;
    }
}

// Handle resend verification email
if (isset($_POST['resend_verification'])) {
    try {
        // Validate CSRF token
        if (!Validator::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Invalid request. Please try again.');
        }
        
        // Validate email
        $email = Validator::validateEmail($_POST['email'] ?? '');
        
        if (!$email) {
            $errors[] = "Please enter a valid email address";
        } else {
            // Attempt to resend verification email
            if (EmailVerification::resendVerification($email)) {
                $success = true;
                $verificationStatus = 'success';
                $verificationMessage = "Verification email has been sent to your email address. Please check your inbox and click the verification link.";
            } else {
                $verificationStatus = 'error';
                $verificationMessage = "Unable to send verification email. Please make sure the email address is correct and try again.";
            }
        }
    } catch (Exception $e) {
        error_log("Email verification error: " . $e->getMessage());
        $verificationStatus = 'error';
        $verificationMessage = "An error occurred. Please try again later.";
    }
}

// Generate CSRF token
$csrfToken = Validator::generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Verification - Migration.ng Portal</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="../assets/favicon/favicon.ico">
    <link rel="icon" type="image/png" sizes="16x16" href="../assets/favicon/favicon-16x16.png">
    <link rel="icon" type="image/png" sizes="32x32" href="../assets/favicon/favicon-32x32.png">
    <link rel="apple-touch-icon" sizes="180x180" href="../assets/favicon/apple-touch-icon.png">
    <link rel="manifest" href="../assets/favicon/site.webmanifest">
    
    <script src="https://ibex.bg/umm/assets/js/tailwind.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-50 min-h-screen flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
    <div class="max-w-md w-full space-y-8">
        <!-- Logo and Header -->
        <div class="text-center">
            <a href="../" class="inline-block">
                <img src="../assets/images/m..png" alt="Migration.ng" class="h-16 mx-auto">
            </a>
            <h2 class="mt-6 text-3xl font-extrabold text-gray-900">
                Email Verification
            </h2>
            <p class="mt-2 text-sm text-gray-600">
                Verify your email address to activate your account
            </p>
        </div>

        <!-- Verification Status -->
        <?php if (!empty($verificationStatus)): ?>
            <div class="bg-white py-8 px-6 shadow rounded-lg sm:px-10">
                <?php if ($verificationStatus === 'success'): ?>
                    <div class="text-center">
                        <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-green-100 mb-4">
                            <i class="fas fa-check text-green-600 text-xl"></i>
                        </div>
                        <h3 class="text-lg font-medium text-gray-900 mb-2">Verification Successful!</h3>
                        <p class="text-sm text-gray-600 mb-6"><?php echo htmlspecialchars($verificationMessage); ?></p>
                        <a href="login.php" class="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            <i class="fas fa-sign-in-alt mr-2"></i>
                            Login to Your Account
                        </a>
                    </div>
                <?php else: ?>
                    <div class="text-center">
                        <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100 mb-4">
                            <i class="fas fa-exclamation-triangle text-red-600 text-xl"></i>
                        </div>
                        <h3 class="text-lg font-medium text-gray-900 mb-2">Verification Failed</h3>
                        <p class="text-sm text-gray-600 mb-6"><?php echo htmlspecialchars($verificationMessage); ?></p>
                        
                        <?php if ($showResendForm): ?>
                            <a href="#resend-form" class="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                <i class="fas fa-paper-plane mr-2"></i>
                                Resend Verification Email
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <!-- Default Form when no verification status -->
            <div class="bg-white py-8 px-6 shadow rounded-lg sm:px-10">
                <div class="text-center mb-6">
                    <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-blue-100 mb-4">
                        <i class="fas fa-envelope text-blue-600 text-xl"></i>
                    </div>
                    <h3 class="text-lg font-medium text-gray-900">Email Verification</h3>
                    <p class="text-sm text-gray-600 mt-1">Enter your email address to receive a verification link</p>
                </div>
                
                <form method="POST" action="">
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                    
                    <div class="mb-6">
                        <label for="email" class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                        <input
                            type="email"
                            name="email"
                            id="email"
                            required
                            class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Enter your email address"
                        />
                    </div>

                    <div>
                        <button
                            type="submit"
                            name="resend_verification"
                            class="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                        >
                            <i class="fas fa-paper-plane mr-2"></i>
                            Send Verification Email
                        </button>
                    </div>
                </form>
            </div>
        <?php endif; ?>

        <!-- Resend Verification Form -->
        <?php if ($showResendForm): ?>
            <div class="bg-white py-8 px-6 shadow rounded-lg sm:px-10" id="resend-form">
                <div class="text-center mb-6">
                    <h3 class="text-lg font-medium text-gray-900">Resend Verification Email</h3>
                    <p class="text-sm text-gray-600 mt-1">Enter your email address to receive a new verification link</p>
                </div>
                
                <form method="POST" action="">
                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                    
                    <div class="mb-6">
                        <label for="email" class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                        <input
                            type="email"
                            name="email"
                            id="email"
                            required
                            class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                            placeholder="Enter your email address"
                        />
                    </div>

                    <div>
                        <button
                            type="submit"
                            name="resend_verification"
                            class="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                        >
                            <i class="fas fa-paper-plane mr-2"></i>
                            Send Verification Email
                        </button>
                    </div>
                </form>
            </div>
        <?php endif; ?>

        <!-- Back to Login -->
        <div class="text-center">
            <a href="login.php" class="text-blue-600 hover:text-blue-500 text-sm">
                <i class="fas fa-arrow-left mr-1"></i>
                Back to Login
            </a>
        </div>

        <!-- Help Section -->
        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-info-circle text-blue-400"></i>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-blue-800">Need Help?</h3>
                    <div class="mt-2 text-sm text-blue-700">
                        <p>If you're having trouble verifying your email, please contact our support team at <a href="mailto:services@migration.ng" class="underline">services@migration.ng</a></p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html> 